/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.deviceio.uart;

import jdk.dio.uart.UART;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.AbstractPollingDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEndpoint;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEvent;
import oracle.iot.messaging.IoTResource;

import javax.inject.Inject;
import java.io.*;
import java.lang.Override;
import java.time.Instant;
import java.time.Duration;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Simulated Uart Thermometer Endpoint class.
 */
@IoTDeviceEndpoint
public class UartThermometerEndpoint extends AbstractPollingDeviceEndpoint implements ThermometerEndpoint {

    /** localized output messages */
    private ResourceBundle messages;

    private SimpleReadOnlyDeviceAttribute<Float> temperature;

    private Logger logger;
    private BufferedReader serialBufferedReader;
    private BufferedWriter serialBufferedWriter;
    private String hardwareId;

    /**
     * This variable is a temperature reading that was read during poll method
     */
    private String temp = "";

    /**
     * @param logger logger associated with this endpoint
     */
    @Inject
    public UartThermometerEndpoint(Logger logger) {
        this.logger = logger;
        messages = ResourceBundle.getBundle("Messages");
        logger.log(Level.INFO, "Created new uart thermometer endpoint.");
    }

    /**
     * Initialize this endpoint with a serial port input stream
     *
     * @param hardwareId        a globally unique hardware ID.
     * @param serialInputStream Physical address of the device on the sample protocol.
     */
    void init(String hardwareId, InputStream serialInputStream, OutputStream serialOutputStream) {
        this.hardwareId = hardwareId;
        serialBufferedReader = new BufferedReader(new InputStreamReader(serialInputStream));
        serialBufferedWriter = new BufferedWriter(new OutputStreamWriter(serialOutputStream));
        /* Set the polling rate to 10 seconds. Every 10 seconds, the poll method will be called to read the data from the buffer */
        setPollingRate(Duration.ofMillis(10000));

        temperature = new SimpleReadOnlyDeviceAttribute<>(
                this, "temperature", getEndpointContext().getEventService());
        logger.info("Starting UART thermometer listener...");
    }

    /**
     * Send the UI input prompt across the UART device
     */
    private void writePrompt() {
        try {
            //serialBufferedWriter.write(new char[]{'\n', '\r', '>'});
            serialBufferedWriter.write(">");
            serialBufferedWriter.flush();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Processes a reading from the UART thermometer and sends the measurement message
     * to the server.
     *
     * @param temp the temperature data read from the UART thermometer
     */

    private void processTemperatureReading(String temp) {
        float t;
        try {
            t = Float.parseFloat(temp);
            System.out.printf(messages.getString("temperature"), getEndpointContext().getEndpointId(),t);

            getEndpointContext().getMessagingService().submit(new ThermometerEvent.Builder(this)
                    .temperature(t)
                    .build().toDataMessage());
            temperature.notifyValueUpdated(t);
        } catch(Exception e) {
            logger.log(Level.FINEST, "Invalid temperature value: {0}", temp);
        }
    }

    /**
     * Method that is invoked to do the actual polling.
     *
     * The poll rate has been set to once every 10 seconds, which means the poll() will be called every 10 seconds,
     * when the poll happens, one '>' would show up, then readline() get called, and user will have 5 seconds to type
     * a number, if the user do not type anything, he should wait for next poll() and readline() get called. A new '>'
     * will not be printed out until it recieve the data from the last poll().
     */
    @Override
    protected void poll() {

        if( temp!=null ) {
            writePrompt();
            temp = null;
        }

        try {
            temp = serialBufferedReader.readLine();
            logger.fine("Temperature reading:" + temp);
        } catch (IOException e) {
            e.printStackTrace();
        }

        if( temp!=null ) {
            processTemperatureReading(temp);
        }
    }

    /**
     * Device framework resource access method.
     *
     * @return the temperature property for this endpoint
     */
    @IoTResource
    public ReadOnlyDeviceAttribute<Float> temperatureProperty() {
        return temperature;
    }

    @Override
    public Float getTemperature() {
        return temperature.getValue();
    }

}
