/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.eddystonebeacon;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * This class represents an Eddystone Beacon event to be sent to IOTCS.
 */
public class EddystoneBeaconEvent<T extends EddystoneBeaconEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:location:eddystone-tlm-uid";

    private Float ora_rssi = null;
    private Integer ora_txPower = null;
    private Float temperature = null;
    private Float batteryVoltage = null;

    
    public final Float getOra_rssi() {
        return ora_rssi;
    }

    public final Integer getOra_txPower() {
        return ora_txPower;
    }

    public final Float get_batteryVoltage() {
        return batteryVoltage;
    }

    public final Float get_temperature() {
        return temperature;
    }


    public static final EventType<EddystoneBeaconEvent<? extends EddystoneBeaconEndpoint>> REPORT =
            new EventType<>("EDDYSTONEBEACON", DataEvent.ANY);

    protected EddystoneBeaconEvent(final T source,
                                   final EventType<? extends EddystoneBeaconEvent> type,
                                   final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        
        if (ora_rssi != null) builder.dataItem("ora_rssi", ora_rssi);
        if (ora_txPower != null) builder.dataItem("ora_txPower", ora_txPower);
        if (temperature != null) builder.dataItem("temperature", temperature);
        if (batteryVoltage != null) builder.dataItem("batteryVoltage", batteryVoltage);
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);

        if (ora_rssi != null) out.append("ora_rssi", ora_rssi);
        if (ora_txPower != null) out.append("ora_txPower", ora_txPower);
        if (temperature != null) out.append("temperature", temperature);
        if (batteryVoltage != null) out.append("batteryVoltage", batteryVoltage);
        return out;
    }

    @Override
    public EddystoneBeaconEvent<T> copy(final T newSource) {
        EddystoneBeaconEvent event = new EddystoneBeaconEvent(newSource, EddystoneBeaconEvent.REPORT, getTimestamp());
        event.ora_rssi = ora_rssi;
        event.ora_txPower = ora_txPower;
        event.batteryVoltage = batteryVoltage;
        event.temperature = temperature;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("EddystoneBeaconEvent")).toString();
    }

    public static class Builder<T extends EddystoneBeaconEndpoint> {
        T source;
        Instant timestamp;
        private Float ora_rssi = null;
        private Integer ora_txPower = null;
        private Float temperature = -128.0f;
        private Float batteryVoltage = 0.0f;

        public Builder(final T source) { this.source = source; this.timestamp = Instant.now(); }
        public Builder(final T source, final Instant timestamp) { this.source = source; this.timestamp = timestamp;
    }

        public Builder ora_rssi(Float ora_rssi) { this.ora_rssi = ora_rssi; return this; }
        public Builder ora_txPower(Integer ora_txPower) { this.ora_txPower = ora_txPower; return this; }
        public Builder temperature(Float temperature) { this.temperature = temperature; return this; }
        public Builder batteryVoltage(Float batteryVoltage) { this.batteryVoltage = batteryVoltage; return this; }

        public EddystoneBeaconEvent build() {
            EddystoneBeaconEvent event = new EddystoneBeaconEvent(source, EddystoneBeaconEvent.REPORT, timestamp);
            event.ora_rssi = ora_rssi;
            event.ora_txPower = ora_txPower;
            event.temperature = temperature;
            event.batteryVoltage = batteryVoltage;
            return event;
        }
    }
}
