/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.bluetoothle;

import com.oracle.bluetooth.le.ProximityMonitor;
import jdk.bluetooth.BluetoothIOException;
import jdk.bluetooth.RemoteDevice;

import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.Metadata;
import oracle.iot.messaging.IoTResource;

import javax.inject.Inject;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Endpoint implementation for Bluetooth LE Proximity Profile capable devices.
 */
@IoTDeviceEndpoint
public class ProximityEndpoint extends BluetoothLEEndpoint {
    protected RemoteDevice device;
    private String hardwareId;

    private ProximityMonitor.ALERT_LEVEL linkLossLevel;
    private ProximityMonitor.ALERT_LEVEL immediateAlert;
    protected Logger logger;
    protected Metadata metadata;
    protected String address;
    protected ProximityMonitor proximityMonitor;

    @Inject
    public ProximityEndpoint() {
        super();
        linkLossLevel = ProximityMonitor.ALERT_LEVEL.NONE;
        immediateAlert = ProximityMonitor.ALERT_LEVEL.NONE;
    }

    /**
     * Return the current immediate alert level as indicated by the remote device.
     *
     * @return the current immediate alert level
     */
    @IoTResource
    public ProximityMonitor.ALERT_LEVEL getImmediateAlert() {
        return immediateAlert;
    }

    /**
     * Pass an immediate alert level request from the server to the remote device.
     *
     * @param alert the alert level requested
     */
    @IoTResource
    public void setImmediateAlert(ProximityMonitor.ALERT_LEVEL alert) {
        try {
            proximityMonitor.setImmediateAlertLevel(device, alert);
        } catch (BluetoothIOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Return the current link loss level as indicated by the remote device.
     *
     * @return the current link loss level
     */
    @IoTResource
    public ProximityMonitor.ALERT_LEVEL getLinkLossLevel() {
        return linkLossLevel;
    }

    /**
     * Pass an link loss level request from the server to the remote device.
     *
     * @param level the link loss level requested
     */
    @IoTResource
    public void setLinkLossLevel(ProximityMonitor.ALERT_LEVEL level) {
        try {
            proximityMonitor.setLinkLossLevel(device, level);
        } catch (BluetoothIOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Set device and register for alert change notifications from proximity device
     *
     * @param device the remote proximity profile device
     */
    public void connect(RemoteDevice device) {
        try {
            this.device = device;
            logger.log(Level.FINE, "Connecting to device: " + device.getAddress());

            proximityMonitor = ProximityMonitor.getInstance();
            proximityMonitor.connect(device, (Map<String, ProximityMonitor.ALERT_LEVEL> data) -> alertLevelChanged(data));
            proximityMonitor.getLinkLossLevel(device);
            proximityMonitor.getImmediateAlertLevel(device);

        } catch (BluetoothIOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Disconnect the proximity listener from the remote device
     */
    public void disconnect() {
        try {
            proximityMonitor.disconnect(device);
        } catch (BluetoothIOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Initializes  metadata associated with this endpoint.
     *
     * @param hardwareId a globally unique ID.
     * @param metadata   the IoT framework metadata associated with this device
     * @param address    the bluetooth address for the associated connected device
     * @param logger     the logger to use for this device
     */
    public void init(String hardwareId, Metadata metadata, String address, Logger logger) {
        this.metadata = metadata;
        this.address = address;
        this.hardwareId = hardwareId;
        this.logger = logger;
    }

    /**
     * Process notification from proximity device
     *
     * @param alerts updated alert level information
     */
    public void alertLevelChanged(Map<String, ProximityMonitor.ALERT_LEVEL> alerts) {

        for (String key : alerts.keySet()) {
            logger.log(Level.FINE, "==================================================");
            logger.log(Level.FINE, "Alert received from: " + device.getAddress() + " at " + System.currentTimeMillis());
            logger.log(Level.FINE, "Key: " + key + " alert: " + alerts.get(key));
            logger.log(Level.FINE, "==================================================");

            if(key.equals(ProximityMonitor.KEY_LINK_LOSS)) {
                linkLossLevel = alerts.get(key);
            }
            if(key.equals(ProximityMonitor.KEY_IMMEDIATE_ALERT)) {
                immediateAlert = alerts.get(key);
            }
        }
    }

}
