/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.app.bluetooth;

import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;
import oracle.iot.device.DeviceProviderService;
import com.oracle.iot.sample.daf.type.bloodpressuremonitor.BloodPressureMonitorEvent;
import com.oracle.iot.sample.daf.type.bodyweightscale.BodyWeightScaleEvent;
import com.oracle.iot.sample.daf.type.pulseoximeter.PulseOximeterEvent;
import com.oracle.iot.sample.daf.type.heartratemonitor.HeartRateMonitorEvent;
import oracle.iot.event.EventHandlerReference;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.util.ArrayList;
import java.util.ResourceBundle;

/**
 * Application that works with sample PHD medical device adapters.
 */
@IoTDeviceApplication
public class BluetoothHealthApplication extends AbstractApplication {

    /** event handler registration references */
    private ArrayList<EventHandlerReference> eventHandlers = new ArrayList<>();

    /** localized output messages */
    private ResourceBundle messages;

    /**
     * The MessagingService to handle the body weight threshold alert.
     */
    private MessagingService messagingService;

    /**
     * Creates a new instance of the {@code BluetoothApplication}
     * @param deviceProvider Device provider used to obtain references
     *                       to device endpoints
     * @param messagingService messaging service to communicate alerts to server
     */
    @Inject
    public BluetoothHealthApplication(DeviceProviderService deviceProvider, MessagingService messagingService) {
        this.messagingService = messagingService;
        messages = ResourceBundle.getBundle("Messages");
        System.out.printf(messages.getString("sample_created"));;
    }

    @Override
    protected void start() {
        System.out.printf(messages.getString("sample_started"));

        synchronized(eventHandlers) {

            eventHandlers.add(getEndpointContext().getEventService().register(BodyWeightScaleEvent.REPORT,
                (event) -> {
                    // Print the event to console
                    System.out.printf(messages.getString("weight_measured"), event.getSource().getId(),
                            event.getBodyWeight());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(
                            event.toDataMessage());
                    String.class.getName();

                }));
            eventHandlers.add(getEndpointContext().getEventService().register(BloodPressureMonitorEvent.REPORT,
                (event) -> {
                    // Print the event to console
                    System.out.printf(messages.getString("bp_measured"), event.getSource().getId(), event.getHeartRate(),
                            event.getDiastolicPressure(), event.getSystolicPressure());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(
                            event.toDataMessage());
                }));


            // Register for pulse oximeter events
            eventHandlers.add(getEndpointContext().getEventService().register(PulseOximeterEvent.REPORT,
                    (event) -> {
                        
                        System.out.printf(messages.getString("o2_measured"), event.getSource().getId(),
                                ((PulseOximeterEvent) event).getOxygenSaturation(), event.getHeartRate());
                        
                        getEndpointContext().getMessagingService().submit(
                                event.toDataMessage());
                    }));
            
            // Register for heart rate events
            eventHandlers.add(getEndpointContext().getEventService().register(HeartRateMonitorEvent.REPORT,
                (event) -> {

                     System.out.printf(messages.getString("hr_measured"), event.getSource().getId(),
                                event.getHeartRate());

                    // Send corresponding data message to the server
                    getEndpointContext().getMessagingService().submit(
                            event.toDataMessage());
                }));
        }
    }


    @Override
    protected void stop() {
        super.stop();

        // unregister our event handlers
        synchronized (eventHandlers) {
            for (EventHandlerReference handler : eventHandlers) {
                handler.unregister();
            }
        }
    }
}
