/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.bluetooth;



import jdk.bluetooth.BluetoothCallback;
import jdk.bluetooth.BluetoothChannel;
import jdk.bluetooth.BluetoothIOException;
import jdk.bluetooth.RemoteDevice;
import jdk.bluetooth.BluetoothStateException;
import jdk.bluetooth.health.HealthSink;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.Metadata;
import com.oracle.iot.sample.daf.adapter.health.PersonalHealthEndpoint;

import java.io.IOException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.HashMap;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Device adapter for {@code PHD} based Bluetooth medical devices
 * <p>
 *     This device adapter handles Bluetooth HDP Personal Health Devices.
 * </p>
 */
public abstract class AbstractBluetoothHealthAdapter extends AbstractDeviceAdapter {

    protected Logger logger;

    private HashMap<Integer, Class> deviceClasses = new HashMap<>();
    private HashMap<Integer, HealthSink> hdpSinks = new HashMap<>();

    /**
     * Creates a new instance of the {@code BluetoothHealthAdapter}.
     * @param logger Logger associated with this endpoint.
     */
    public AbstractBluetoothHealthAdapter(Logger logger) {
        this.logger = logger;
    }

    /**
     * Stops the device adapter.
     * Calls the {@code super.stop()} method which takes care of stopping
     * and unregistering all devices created by this adapter. Unregisters
     * all hdp sinks
     */
    @Override
    protected void stop() throws Exception {
        super.stop();
        Set<Integer> types = hdpSinks.keySet();
        for(Integer type: types)
            unregisterDeviceClass(type);
    }

    /**
     * Registers a PersonalHealthEndpoint with the device framework.
     * This registration performs no additional communication with the endpoint
     * when complete.
     *
     * @param bhe the personal health endpoint to register with framework. Must not be {@code null}.
     */
    private void registerHealthEndpoint(PersonalHealthEndpoint bhe) {
        final ObservableFuture<PersonalHealthEndpoint> future = registerDevice(bhe.getHardwareId(), bhe.getMetadata(),
                PersonalHealthEndpoint.class, null);
        future.addListener((observable) -> {
            logger.log(Level.INFO, "Personal health endpoint registered:" + bhe.getAddress());
        });
    }

    /**
     * Associates a bluetooth health endpoint implementation class with a IEEE 11073 device type.
     *
     * @param type the device type to associate with a device implementation class
     * @param cls the device implementation class
     */
    protected void registerDeviceClass(int type, Class cls)
    {
        deviceClasses.put(type, cls);
        synchronized(hdpSinks) {
            if (hdpSinks.get(type) == null) {
                initDeviceSink(type);
            }
        }
    }

    protected void unregisterDeviceClass(int type)
    {
        deviceClasses.remove(type);
        HealthSink sink = null;
        synchronized(hdpSinks) {
            sink = hdpSinks.remove(type);
            logger.log(Level.FINE, "unregistering sink for device type: " + type);
            if (sink != null) {
                try {
                    sink.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * This is an HDP connection listener implementation. Each listener implementation is associated with a device type
     * (e.g. PulseOximeter), and there is a single implementation for each type.
     * On connections, first check to see if the connecting device is already in the local adapter's database. If not,
     * gather metadata and register the device with the framework.
     *
     * Once the device registration process is started, create the Phd protocol
     * handler and begin the Phd connection/data exchange process.
     */
    class HDPListener implements BluetoothCallback {
        int type;
        HDPListener(int _type)
        {
            type = _type;
        }

        @Override
        public void channelConnected(BluetoothChannel bluetoothChannel) {

            // retrieve the endpoint from our local registry by address.
            String addr = bluetoothChannel.getRemoteEndpoint().getDevice().getAddress();
            PersonalHealthEndpoint bhe = EndpointRegistry.getInstance().getEndpoint(addr);

            logger.log(Level.INFO, "Received bluetooth connection from  <" + addr + ">");
            // if the bhe is already in our local database, have the endpoint start PHD communication
            if(bhe != null)
            {
                // notify the endpoint that a connection has been received
                try {
                    bhe.channelConnect(bluetoothChannel.getInputStream(), bluetoothChannel.getOutputStream());
                } catch (IOException e) {
                    logger.log(Level.WARNING, "Exception attempting to connect health endpoint: " + e);
                }
            }
            else
            {
                // remote device is not yet locally known - register device before processing connection
                addBluetoothHealthEndpoint(bluetoothChannel);
            }
        }

        @Override
        public void channelDisconnected(BluetoothChannel bluetoothChannel) {
            String addr = bluetoothChannel.getRemoteEndpoint().getDevice().getAddress();
            PersonalHealthEndpoint bhe = EndpointRegistry.getInstance().getEndpoint(addr);

            logger.log(Level.INFO, "Received bluetooth disconnection from  <" + addr + ">");
            if(bhe != null)
            {
                // notify device of disconnection
                bhe.channelDisconnect();
            }
        }

        /**
         * Given an HDP connection for a device that is not yet locally registered, inspect the available bluetooth
         * metadata and create a type specific bluetooth endpoint and register the endpoint with the framework.
         * @param bluetoothChannel, must not be {@code null}
         */
        private void addBluetoothHealthEndpoint(BluetoothChannel bluetoothChannel) {
            // gather metadata
            RemoteDevice device = (RemoteDevice)bluetoothChannel.getRemoteEndpoint().getDevice();
            String name = device.getName();
            String addr = device.getAddress();
            String manufacturer = Integer.toString(device.getManufacturerID());

            if(manufacturer.equals("-1")) {
                manufacturer = "65535"; //This value is reserved as the default vendor ID when no Device ID service record is present in a remote device.
            }


            logger.log(Level.FINE, "registering BluetoothHealthEndpoint: name[" + name + "] addr[" + addr + "] "
                    + "manufacturer[" + manufacturer + "] productId:[" + device.getProductID() + "] vendorId:["
                    + device.getVendorSourceID() + "]");

            // instantiate metadata class
            // As we do not have a way to retrieve the serial number from the PHD device at the time
            // of device registration, we use the protocol ID as the serial number.
            Metadata metadata = Metadata.builder().
                protocol("bluetooth").
                protocolDeviceClass("hdp").
                protocolDeviceId(addr).
                manufacturer(manufacturer).
                deviceClass(PersonalHealthEndpoint.getClassName(type)).
                serialNumber(addr).build();

            AbstractBluetoothHealthAdapter.this.create(metadata, addr, type, bluetoothChannel);
        }
    }

    /**
     * Creates the HDP application sync based on PHD device type
     *
     * @param type the IEEE-11073 device type to listen for
     */
    protected void initDeviceSink(int type) {
        HealthSink application =  AccessController.doPrivileged((PrivilegedAction<HealthSink>) () -> {
            try {
                return HealthSink.newSink(type, "HealthSink for " + type, new HDPListener(type));
            } catch(BluetoothIOException | BluetoothStateException e) {
                e.printStackTrace();
                return null;
            }
        });

        if (application != null) {
            hdpSinks.put(type, application);
            logger.log(Level.FINE, "HDP sink created: " + application + " for device type: " + type);
        }
    }

    /**
     * Registers a connected device with the device framework and, if successful, initiates PHD connection.
     *
     * @param metadata device framework metadata to associated with connected device. Must not be {@code null}.
     * @param addr bluetooth address for the connected device
     * @param type the IEEE 11073 device type
     * @param bluetoothChannel the connected HDP channel associated with this device. Must not be {@code null}
     */
    public void create(Metadata metadata, String addr, int type, BluetoothChannel bluetoothChannel)
    {
        Class deviceClass;
        logger.log(Level.INFO, "Creating [" + type + "] framework device");
        deviceClass = deviceClasses.get(type);
        if(deviceClass == null)
        {
            logger.log(Level.WARNING, "Bluetooth Health Adapter unable to find device class for type : " + type);
            return;
        }
        ObservableFuture<PersonalHealthEndpoint> future = registerDevice(addr, metadata, deviceClass, null);
        future.addListener((observable) -> {
            try {
                PersonalHealthEndpoint endpoint = future.get();
                logger.log(Level.INFO, "Personal health endpoint registered: " + endpoint.getId());
                EndpointRegistry.getInstance().addEndpoint(addr, endpoint);
                endpoint.init(addr, metadata, addr, type, logger);
                try {
                    endpoint.channelConnect(bluetoothChannel.getInputStream(), bluetoothChannel.getOutputStream());
                } catch (IOException e) {
                    logger.log(Level.WARNING, "Exception attempting to connect health endpoint: " + e);
                }
            } catch (final Exception e) {
                e.printStackTrace();
            }
        });
    }
}
