/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.app.beacon;

import com.oracle.iot.sample.daf.type.ibeacon.IBeaconEvent;
import com.oracle.iot.sample.daf.type.eddystonebeacon.EddystoneBeaconEvent;
import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;
import oracle.iot.event.EventHandlerReference;
import oracle.iot.event.EventService;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.util.ArrayList;
import java.util.Date;

/**
 * Application that works with sample iBeacon device adapters.
 */
@IoTDeviceApplication
public class BeaconApplication extends AbstractApplication {
    private ArrayList<EventHandlerReference> eventHandlers = new ArrayList<>();
    private MessagingService messagingService;


    @Inject
    public BeaconApplication(MessagingService messagingService) {
        this.messagingService = messagingService;
    }

    @Override
    protected void start() {
        synchronized (eventHandlers) {
            EventService eventService = getEndpointContext().getEventService();

            // Register for iBeacon events.
            eventHandlers.add(eventService.register(IBeaconEvent.REPORT,
                (event) -> {
                    Float ora_rssi = event.getOra_rssi();
                    Integer ora_txPower = event.getOra_txPower();

                    // Print the event to console
                    StringBuilder outputMsg = new StringBuilder(
                        new Date().toString())
                        .append(" : ")
                        .append(getId());

                    // Check if ora_rxPower is set since it does not change often.
                    if (ora_txPower != null) {
                        outputMsg.append("\", ora_txPower\" = ").append(ora_txPower);
                    }
                    outputMsg.append("\", ora_rssi\" = ").append(ora_rssi);

                    System.out.println(outputMsg.toString());

                    // Send corresponding data message for iBeacon to the server.
                    messagingService.submit(event.toDataMessage());
                }));

            // Register for EddystoneBeacon events.
            eventHandlers.add(eventService.register(EddystoneBeaconEvent.REPORT,
                    (event) -> {
                        Float ora_rssi = event.getOra_rssi();
                        Integer ora_txPower = event.getOra_txPower();
                        Float temperature = event.get_temperature();
                        Float batteryVoltage = event.get_batteryVoltage();

                        // Print the event to console
                        StringBuilder outputMsg = new StringBuilder(
                                new Date().toString())
                                .append(" : ")
                                .append(getId());

                        // Check if ora_rxPower is set since it does not change often.
                        if (ora_txPower != null) {
                            outputMsg.append("\", ora_txPower\" = ").append(ora_txPower);
                        }

                        outputMsg.append("\", ora_rssi\" = ").append(ora_rssi)
                                .append("\", temperature\" = ").append(temperature)
                                .append("\", batteryVoltage\" = ").append(batteryVoltage);

                        System.out.println(outputMsg.toString());

                        // Send corresponding data message for Eddystone beacon to the server.
                        messagingService.submit(event.toDataMessage());
                    }));
        }
    }

    @Override
    protected void stop() {
        super.stop();

        // Unregister our event handlers.
        synchronized (eventHandlers) {
            for (EventHandlerReference handler : eventHandlers) {
                handler.unregister();
            }
        }
    }
}
