/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import oracle.iot.device.Metadata;
import oracle.iot.messaging.IoTResource;
import oracle.iot.module.IoTExport;

/**
 *  This class represents the Meta-data associated with the iBeacon devices.
 */
@IoTExport
public interface IBeaconMetadata extends Metadata {
    /** This represents the Major number of the iBeacon device. */
    @IoTResource
    String getMajor();

    /** This represents the Minor number of the iBeacon device. */
    @IoTResource
    String getMinor();

    /** This represents the UUID of the iBeacon device. */
    @IoTResource
    String getUuid();


    static Builder<?> builder() {
        return new Builder();
    }

    class Default extends Metadata.Default implements IBeaconMetadata {
        private static final String MINOR = "major";
        private static final String MAJOR = "minor";
        private static final String UUID  = "uuid";

        protected Default(IBeaconMetadata.Builder<?> builder) {
            super(builder);
        }

        @Override
        public String getMajor() {
            return (String) getAttribute(MAJOR);
        }

        @Override
        public String getMinor() {
            return (String) getAttribute(MINOR);
        }

        @Override
        public String getUuid() {
            return (String) getAttribute(UUID);
        }
    }

    class Builder<T extends Builder<T>> extends Metadata.Builder<T> {
        protected Builder() {
        }

        @Override
        public IBeaconMetadata build() {
            return new Default(this);
        }

        @SuppressWarnings("unchecked")
        public final T major(String major) {
            attributes.put(Default.MAJOR, major);
            return (T) this;
        }

        @SuppressWarnings("unchecked")
        public final T minor(String minor) {
            attributes.put(Default.MINOR, minor);
            return (T) this;
        }

        @SuppressWarnings("unchecked")
        public final T uuid(String uuid) {
            attributes.put(Default.UUID, uuid);
            return (T) this;
        }
    }
}
