/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import com.oracle.iot.sample.daf.type.ibeacon.IBeaconEndpoint;
import com.oracle.iot.sample.daf.type.ibeacon.IBeaconEvent;

import com.oracle.bluetooth.le.AdvertisementDataParser;
import com.oracle.bluetooth.le.AdvertisementDataParser.AdvertisementDataFormat;
import com.oracle.bluetooth.le.IBeaconAdvertisementData;
import com.oracle.bluetooth.le.Scanner;

import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;

import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Arrays;

import javax.inject.Inject;
import javax.xml.ws.EndpointContext;

import jdk.bluetooth.BluetoothException;
import jdk.bluetooth.RemoteDevice;

/**
 * Represents an iBeacon device in the IoT CS.  Registers to receive RSSI value changes of the iBeacon
 * and sends them to IoT CS.
 */
@IoTDeviceEndpoint
public class IBeaconEndpointImpl extends AbstractDeviceEndpoint implements IBeaconEndpoint {
    private final EventService eventService;
    private Scanner scanner = null;

    /** The RSSI (Received Signal Strength Indicator) and TxPower (Transmission Power) values. */
    private Float ora_rssiValue;
    private Integer ora_txPowerValue;
    private RemoteDevice remoteDevice;

    /** Device attributes for RSSI and TxPower. */
    private SimpleReadOnlyDeviceAttribute<Float> ora_rssi;
    private SimpleReadOnlyDeviceAttribute<Integer> ora_txPower;
    private IBeaconAdvertisementData beaconAdvData;
    private RssiSmoother rssiSmoother;


    @Inject
    public IBeaconEndpointImpl(EventService eventService, Logger logger) {
        super();
        this.eventService = eventService;
        this.rssiSmoother = new RssiSmoother();
        ora_rssi = new SimpleReadOnlyDeviceAttribute<>(this, "ora_rssi", eventService, () -> ora_rssiValue);
        ora_txPower = new SimpleReadOnlyDeviceAttribute<>(this, "ora_txPower", eventService, () -> ora_txPowerValue);
        try {
            scanner = Scanner.getInstance();
            beaconAdvData = null;
        }  catch (BluetoothException ex) {
            ex.printStackTrace();
        }
        ora_txPowerValue = 0;
    }

    /**
     * This method initiailizes this IBeacon endpoint and starts the RSSI notification for the iBeacon.
     */
    public void init(RemoteDevice remoteDevice, IBeaconAdvertisementData advData) {
        this.remoteDevice = remoteDevice;
        this.beaconAdvData = advData;
        rssiSmoother.setTxPower(advData.getTxPower());
        // Subscribe for RSSI notification for this iBeacon.
        scanner.subscribeRssiNotification(remoteDevice, new IBeaconConsumer(this));
    }

    @Override public ReadOnlyDeviceAttribute<Float> ora_rssiProperty() { return ora_rssi; }
    @Override public ReadOnlyDeviceAttribute<Integer> ora_txPowerProperty() { return ora_txPower; }

    /**
     * This is a private class for consuming the RSSI value changes for the iBeacon device.
     */
    private class IBeaconConsumer implements BiConsumer<RemoteDevice, Integer> {
        private IBeaconEndpointImpl iBeaconEndpoint;

        /**
         * This method consumes the RSSI value changes for the iBeacon device.
         */
        @Override
        public void accept(RemoteDevice remoteDevice, Integer rssiValue) {

            // pass the new rssi value to RssiSmoother for smoothing.
            rssiSmoother.setRssi(rssiValue);

            // Get the smoothed out rssi value.
            int rssiVal = rssiSmoother.getRssi();

            // Send the smoothed RSSI value to IbeaconEvent.
            IBeaconEvent.Builder iBeaconEventBuilder = new IBeaconEvent.Builder(iBeaconEndpoint)
                .ora_rssi((float) rssiVal);

            if (beaconAdvData != null) {
                // Set txPower attribute only it changed from last time.
                int currentTxPower = beaconAdvData.getTxPower();
                if (ora_txPowerValue != currentTxPower) {
                    ora_txPowerValue = currentTxPower;
                    iBeaconEventBuilder.ora_txPower(beaconAdvData.getTxPower());
                }
            }

            // Send the iBeaconEvent to the application which will post it to IoT CS.
            eventService.fire(iBeaconEventBuilder.build());
        }

        public IBeaconConsumer(IBeaconEndpointImpl iBeaconEndpoint) {
            this.iBeaconEndpoint = iBeaconEndpoint;
        }
    }
}
