/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import com.oracle.iot.sample.daf.type.eddystonebeacon.EddystoneBeaconEndpoint;
import com.oracle.iot.sample.daf.type.eddystonebeacon.EddystoneBeaconEvent;

import com.oracle.bluetooth.le.AdvertisementDataParser;
import com.oracle.bluetooth.le.AdvertisementDataParser.AdvertisementDataFormat;
import com.oracle.bluetooth.le.EddystoneUidData;
import com.oracle.bluetooth.le.EddystoneTelemetryData;
import com.oracle.bluetooth.le.Scanner;

import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;

import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.inject.Inject;
import javax.xml.ws.EndpointContext;

import jdk.bluetooth.BluetoothException;
import jdk.bluetooth.RemoteDevice;

/**
 * Represents an Eddystone beacon device in the IoT CS.  Registers to receive RSSI/telemetry data changes
 * for Eddystone beacon devices and sends them to IoT CS.
 */
@IoTDeviceEndpoint
public class EddystoneBeaconEndpointImpl extends AbstractDeviceEndpoint implements EddystoneBeaconEndpoint {
    private final EventService eventService;
    private Scanner scanner = null;

    /** The RSSI (Received Signal Strength Indicator) and TxPower (Transmission Power) values. */
    private Float ora_rssiValue;
    private int ora_txPowerValue;
    private Float batteryVoltageValue;
    private Float temperatureValue;
    private RemoteDevice remoteDevice;

    /** Device attributes for RSSI, TxPower, Battery Voltage and Temperature. */
    private SimpleReadOnlyDeviceAttribute<Float> ora_rssi;
    private SimpleReadOnlyDeviceAttribute<Integer> ora_txPower;
    private SimpleReadOnlyDeviceAttribute<Float> batteryVoltage;
    private SimpleReadOnlyDeviceAttribute<Float> temperature;

    /** Eddystone information objects */
    private EddystoneUidData eddystoneUidData;
    private EddystoneTelemetryData telemetryData;
    private RssiSmoother rssiSmoother;


    @Inject
    public EddystoneBeaconEndpointImpl(EventService eventService, Logger logger) {
        super();
        this.eventService = eventService;
        rssiSmoother = new RssiSmoother();
        ora_rssi = new SimpleReadOnlyDeviceAttribute<>(this, "ora_rssi", eventService, () -> ora_rssiValue);
        ora_txPower = new SimpleReadOnlyDeviceAttribute<>(this, "ora_txPower", eventService, () -> ora_txPowerValue);
        batteryVoltage = new SimpleReadOnlyDeviceAttribute<>(this, "batteryVoltage", eventService, () -> batteryVoltageValue);
        temperature = new SimpleReadOnlyDeviceAttribute<>(this, "temperature", eventService, () -> temperatureValue);
        eddystoneUidData = null;
        ora_txPowerValue = 0;
    }

    /**
     * This method initiailizes this Eddystone endpoint and starts the RSSI notification for the beacon.
     */
    public void init(RemoteDevice remoteDevice, EddystoneUidData eddystoneUidData, Scanner scanner) {
        this.remoteDevice = remoteDevice;
        this.scanner = scanner;
        rssiSmoother.setTxPower(eddystoneUidData.getTxPower());
        // Subscribe for RSSI notification for this beacon.
        scanner.subscribeRssiNotification(remoteDevice, new EddystoneBeaconConsumer(this));
        // Set the EddystoneUidData object  to get the txPower value.
        this.eddystoneUidData = eddystoneUidData;
    }

    /**
     * This method is called from BeaconDeviceAdapter class to update the new telemtry data received for this beacon.
     */
    public  void updateTelemetryData(EddystoneTelemetryData telemetryData) {
        this.telemetryData = telemetryData;
    }

    @Override public ReadOnlyDeviceAttribute<Float> ora_rssiProperty() { return ora_rssi; }
    @Override public ReadOnlyDeviceAttribute<Integer> ora_txPowerProperty() { return ora_txPower; }
    @Override public ReadOnlyDeviceAttribute<Float> batteryVoltageProperty() { return batteryVoltage; }
    @Override public ReadOnlyDeviceAttribute<Float> temperatureProperty() { return temperature; }

    /**
     * This is a private class for consuming the RSSI value changes for the Eddystone endpoint device object.
     */
    private class EddystoneBeaconConsumer implements BiConsumer<RemoteDevice, Integer> {
        private EddystoneBeaconEndpointImpl eddystoneBeaconEndpoint;

        /**
         * This method consumes the RSSI value changes for the Eddystone endpoint device object.         *
         */
        @Override
        public void accept(RemoteDevice remoteDevice, Integer rssiValue) {

            // pass the new rssi value to RssiSmoother for smoothing.
            rssiSmoother.setRssi(rssiValue);

            // Get the smoothed out rssi value.
            int rssiVal = rssiSmoother.getRssi();

            // Create the EddystoneBeaconEvent objet and set the smoothed out RSSI value.
            EddystoneBeaconEvent.Builder eddystoneBeaconEventBuilder =
                    new EddystoneBeaconEvent.Builder(eddystoneBeaconEndpoint).ora_rssi((float) rssiVal);

            // set the txPower value from eddystoneUidData.
            if (eddystoneUidData != null) {
                int currentTxpoweValue = eddystoneUidData.getTxPower();
                // Check if txPower value has changed, and only if it changes set the value in the event.
                if (currentTxpoweValue != ora_txPowerValue) {
                    ora_txPowerValue = currentTxpoweValue;
                    eddystoneBeaconEventBuilder.ora_txPower(ora_txPowerValue);
                }
            }

            // Set the battery Voltage and beacon temerature values from telemetryData.
            if (telemetryData != null) {
                batteryVoltageValue = (float)telemetryData.getBatteryVoltage();
                eddystoneBeaconEventBuilder.batteryVoltage(batteryVoltageValue);

                temperatureValue = (float)telemetryData.getBeaconTemperature();
                eddystoneBeaconEventBuilder.temperature(temperatureValue);
            }

            // Send the EddystoneBeaconEvent to BeaconApplication which will post it to IOTCS.
            eventService.fire(eddystoneBeaconEventBuilder.build());
        }
        /**
         * This function sets the eddystoneBeaconEndpoint object to be used by RSSI consumer.
         */
        public EddystoneBeaconConsumer(EddystoneBeaconEndpointImpl eddystoneBeaconEndpoint) {
            this.eddystoneBeaconEndpoint = eddystoneBeaconEndpoint;
        }
    }

}
