/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import java.util.Date;
import java.util.Hashtable;
import java.util.IllegalFormatException;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.inject.Inject;

import com.oracle.bluetooth.le.AdvertisementDataParser;
import com.oracle.bluetooth.le.AdvertisementDataParser.AdvertisementDataFormat;
import com.oracle.bluetooth.le.IBeaconAdvertisementData;
import com.oracle.bluetooth.le.EddystoneUidData;
import com.oracle.bluetooth.le.EddystoneTelemetryData;
import com.oracle.bluetooth.le.Scanner;

import jdk.bluetooth.BluetoothException;
import jdk.bluetooth.LocalDevice;
import jdk.bluetooth.RemoteDevice;

import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.endpoint.EndpointContext;
import oracle.iot.event.EventService;

/**
 * Device adapter for Bluetooth beacons.  The role of this class is to:
 *    - "discover" beacons (both iBeacons and Eddystone beacons).
 *    - Register the beacon with the IoT CS.
 *    - Instantiate an endpoint to handle RSSI and other data from the beacons and send the data to the IoT CS.
 */
@IoTDeviceAdapter
public class BeaconDeviceAdapter extends AbstractDeviceAdapter {
    private final EventService eventService;
    /** The local Bluetooth adapter. */
    private LocalDevice defaultLocalDevice = null;
    private final Logger logger;
    private final String endpointId;
    private Scanner scanner = null;
    private AdvertisementDataParser parser = null;
    private Hashtable<RemoteDevice, EddystoneBeaconEndpointImpl> mapEddystoneEndpoints= null;

    @Inject
    public BeaconDeviceAdapter(EndpointContext endpointContext) {
        this.endpointId = endpointContext.getEndpointId();
        this.logger = endpointContext.getLogger();
        this.eventService = endpointContext.getEventService();
        this.mapEddystoneEndpoints = new Hashtable<RemoteDevice, EddystoneBeaconEndpointImpl>();
        logger.log(Level.INFO, "BeaconDeviceAdapter constructor created new Beacon device adapter.");
    }

    /**
     * Registers the iBeacon with the IoT CS.
     * @param remoteDevice the Bluetooth RemoteDevice which represents the discovered iBeacon.
     * @param advData the IBeaconAdvertisementData object which represents the advertiisement data of iBeacon.
     */
    private void registerMyDevice(RemoteDevice remoteDevice, final IBeaconAdvertisementData advData) {
        String address = remoteDevice.getAddress();

        IBeaconMetadata iBeaconMetadata = IBeaconMetadata.builder()
            .deviceClass("LE")
            // TODO: Get manufacturer name from device.
            .manufacturer("Estimote")
            .protocol("Bluetooth-LE")
            .protocolDeviceId(address)
            .protocolDeviceClass("iBeacon")
            // TODO: Use: majorVersion + "-" + minorVersion
            .serialNumber("Serial:" + address)
            .build();

        ObservableFuture<IBeaconEndpointImpl> future = registerDevice(
            endpointId + "_" + address,
            iBeaconMetadata,
            IBeaconEndpointImpl.class,
            (iBeaconEndpoint) -> iBeaconEndpoint.init(remoteDevice, advData));

        // Listen for the new device representation being ready.
        future.addListener((observable) -> {
            try {
                IBeaconEndpointImpl iBeaconEndpoint = future.get();
                logger.log(Level.FINE, "iBeacon registered: " + iBeaconEndpoint.getId());
            } catch (final Exception e) {
                e.printStackTrace();
            }
        });
    }

    /**
     * Registers the EddystoneBeacon with the IoT CS.
     * @param remoteDevice the Bluetooth RemoteDevice which represents the discovered Eddystone beacon.
     * @param uidData the EddystoneUidData object which represents the Eddystone UID data of Eddystone beacon.
     */
    private void registerMyDevice(final RemoteDevice remoteDevice, final EddystoneUidData uidData) {
        String address = remoteDevice.getAddress();

        EddystoneBeaconMetadata eddystoneBeaconMetadata = EddystoneBeaconMetadata.builder()
                .deviceClass("LE")
                // TODO: Get manufacturer name from device.
                .manufacturer("Estimote")
                .protocol("Bluetooth-LE")
                .protocolDeviceId(address)
                .protocolDeviceClass("eddystone")
                // TODO: Use: majorVersion + "-" + minorVersion
                .serialNumber("Serial:" + address)
                .build();

        ObservableFuture<EddystoneBeaconEndpointImpl> future = registerDevice(
                endpointId + "_" + address,
                eddystoneBeaconMetadata,
                EddystoneBeaconEndpointImpl.class,
                (eddystoneBeaconEndpoint) -> eddystoneBeaconEndpoint.init(remoteDevice, uidData, scanner));

        // Listen for the new device representation being ready.
        future.addListener((observable) -> {
            try {
                EddystoneBeaconEndpointImpl eddystoneBeaconEndpoint = future.get();
                mapEddystoneEndpoints.put(remoteDevice,eddystoneBeaconEndpoint);
                logger.log(Level.FINE, "Eddystone Beacon registered: " + eddystoneBeaconEndpoint.getId());
            } catch (final Exception e) {
                e.printStackTrace();
            }
        });
    }

    /**
     * Updates the received Eddystone telemetry data to the corresponding EddystoneBeaconEndpointImpl object.
     * @param remoteDevice the Bluetooth RemoteDevice which represents the Eddystone beacon.
     * @param telemetryData the EddystoneTelemtryData object which represents the telemetry data of Eddystone beacon.
     */

    private void updateEddystoneTelemetryData(final RemoteDevice remoteDevice, final EddystoneTelemetryData telemetryData) {
        EddystoneBeaconEndpointImpl enddystoneEndPointImpl = null;
        if (mapEddystoneEndpoints.containsKey(remoteDevice)) {
            enddystoneEndPointImpl = mapEddystoneEndpoints.get(remoteDevice);
            enddystoneEndPointImpl.updateTelemetryData(telemetryData);
        }
    }

    @Override
    protected void start() throws Exception {
        new Thread(()->startBeaconScan()).start();
    }

    public void printBeaconInformation(RemoteDevice remoteDevice, EddystoneUidData eddystoneUidData) {
        System.out.println("!!! Eddystone beacon Device found: " + remoteDevice.getName());
        System.out.println("Namespace Id:" + eddystoneUidData.getNamespaceId());
        System.out.println("Instance Id:" + eddystoneUidData.getInstanceId());
        System.out.println("txPower:" + eddystoneUidData.getTxPower());
    }
    public void printBeaconInformation(RemoteDevice remoteDevice, IBeaconAdvertisementData iBeaconData) {
        System.out.println("!!! IBeacon Device found: " + remoteDevice.getName());
        System.out.println("  UUID:" + iBeaconData.getUUID());
        System.out.println("  Major:" + iBeaconData.getMajorValue());
        System.out.println("  Minor:" + iBeaconData.getMinorValue());
        System.out.println("  txPower:" + iBeaconData.getTxPower());
    }

    /**
     * Starts the scanning process to find beacon devices; both iBeacons and Eddystone devices.
     */
    private void startBeaconScan() {
            parser = AdvertisementDataParser.getInstance();

            // Initialize the scanConsumer to receive scanned beacon devices.
            BiConsumer<RemoteDevice, byte[]> scanConsumer = new BiConsumer<RemoteDevice, byte []>() {
                /**
                 * This method consumes new beacon devices found by the Scanner.
                 */
                @Override
                public void accept(RemoteDevice remoteDevice, byte buffer[]) {
                    AdvertisementDataParser.AdvertisementDataFormat format = parser.parseAdvertisementData(buffer);
                    try {
                        switch (format) {
                            case IBEACON:
                                IBeaconAdvertisementData iBeaconData = new IBeaconAdvertisementData(buffer);
                                printBeaconInformation(remoteDevice, iBeaconData);
                                registerMyDevice(remoteDevice, iBeaconData);
                                break;
                            case EDDYSTONE_UID:
                                EddystoneUidData eddystoneUidData = new EddystoneUidData(buffer);
                                printBeaconInformation(remoteDevice, eddystoneUidData);
                                registerMyDevice(remoteDevice, eddystoneUidData);
                                break;
                            case EDDYSTONE_TELEMETRY:
                                EddystoneTelemetryData eddystoneTelemetryData = new EddystoneTelemetryData(buffer);
                                updateEddystoneTelemetryData(remoteDevice, eddystoneTelemetryData);
                                break;
                            default:
                                System.out.println("Unknown format of Advertisement Data for Device:" + remoteDevice.getName());
                        }
                    } catch (IllegalFormatException ex) {
                        ex.printStackTrace();
                    }
                }
            };

            // Start the scanning process to find beacon devices.
            try {
                scanner = Scanner.getInstance();
                scanner.scan(scanConsumer);
            } catch(BluetoothException ex) {
                ex.printStackTrace();
            }
    }
}
