/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.temperature;

import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEndpoint;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooColdEvent;
import com.oracle.iot.sample.daf.type.temperature.TemperatureSensorTooHotEvent;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Example endpoint implementation of an endpoint which implements the temperature sensor device model
 */
@IoTDeviceEndpoint
public class TemperatureSensorEndpointImpl extends AbstractDeviceEndpoint implements TemperatureSensorEndpoint {
    private static final AtomicBoolean done = new AtomicBoolean(false);
    private final static long OUTPUT_INTERVAL = 2000;
    private static Random random = new Random();
    private boolean isOn;
    private Date startTimeValue;
    private Float   maxTempValue = 0f;
    // The maximum threshold value, initialized to the default in the device model.
    private Integer maxThresholdValue = 70;
    private Float   minTempValue = 0f;
    // The minimum threshold value, initialized to the default in the device model.
    private Integer minThresholdValue = 0;

    // Used to generate a humidity value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;
    // the fluctuating humidity sensor value
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 2f;

    // The current read temperature value.
    private Float tempValue = 0f;
    // The previously read temperature value.
    private Float previousTempValue = 0f;
    private SimpleDeviceAttribute<Integer> minThreshold;
    private SimpleDeviceAttribute<Integer> maxThreshold;
    private SimpleReadOnlyDeviceAttribute<Float> maxTemp;
    private SimpleReadOnlyDeviceAttribute<Float> minTemp;
    private SimpleReadOnlyDeviceAttribute<Date> startTime;
    private SimpleReadOnlyDeviceAttribute<Float> temp;
    private SimpleReadOnlyDeviceAttribute<String> unit;
    private String unitValue = "C";
    private Thread measurementThread = null;

    @Inject
    public TemperatureSensorEndpointImpl(Logger logger) {
        super();
        logger.log(Level.INFO, "Created new temperature sensor endpoint.");
    }

    /**
     * Initializes the device endpoint.
     */
    void init() {
        EventService eventService = getEndpointContext().getEventService();
        startTimeValue = new Date(Instant.now().toEpochMilli());
        // Instantiate the temperature sensor attribute, passing a lambda which simply returns the endpoint's stored
        // temperature value.
        temp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "temperature",
                eventService,
                ()->tempValue);

        // Instantiate the temperature sensor attributes, passing lambdas which simply sets or returns the endpoint's
        // stored unit value.
        unit = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "unit",
                eventService,
                () -> unitValue);

        minThreshold = new SimpleDeviceAttribute<Integer>(
                this,
                "minThreshold ",
                eventService,
                () -> minThresholdValue,
                (s) -> {
                    minThresholdValue = s;
                    System.out.println(new Date().toString() + " : " + getId() + " : Set : \"minThreshold\" = " + minThresholdValue);
                    
                    getEndpointContext().getEventService().fire(new TemperatureSensorEvent.Builder(this)
                            .temp(tempValue)
                            .minThreshold(minThresholdValue)
                            .build()
                    );
                });

        maxThreshold = new SimpleDeviceAttribute<Integer>(
                this,
                "maxThreshold",
                eventService,
                () -> maxThresholdValue,
                (s) -> {
                    maxThresholdValue = s;
                    System.out.println(new Date().toString() + " : " + getId() + " : Set : \"maxThreshold\" = " + maxThresholdValue);

                    getEndpointContext().getEventService().fire(new TemperatureSensorEvent.Builder(this)
                            .temp(tempValue)
                            .maxThreshold(maxThresholdValue)
                            .build()
                    );
                });

        maxTemp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "maxTemp",
                eventService,
                () -> maxTempValue);

        minTemp = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "minTemp",
                eventService,
                () -> minTempValue);

        startTime = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "startTime",
                eventService,
                () -> startTimeValue);
    }

    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> maxTempProperty() {
        return maxTemp;
    }

    @Override
    public DeviceAttribute<Integer> minThresholdProperty() {
        return minThreshold;
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> minTempProperty() {
        return minTemp;
    }

    @Override
    public void power(Boolean isOn) {
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"power\" = " + isOn);
        this.isOn = isOn;
    }

    @Override
    public void reset() {
        // Reset the temperature.
        System.out.println(new Date().toString() + " : " + getId() + " : " + " Action: \"reset\"");
        minTempValue = tempValue;
        maxTempValue = tempValue;
    }

    @Override
    protected void start() throws Exception {
        done.set(false);
        super.start();

        measurementThread = new Thread(() -> {
            isOn = true;

            // TODO: add shutdown logic to this loop
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }

                    if (isOn) {
                        // Generate a temperature value which exceeds the min and max threshold values.  The min and max
                        // threshold values can be found in the TemperatureSensor device model.
                        tempValue = calculateNextValue();

                        // Only send events if maxTemp, minTemp, or temperature has changed.
                        if ((tempValue.intValue() != previousTempValue.intValue())) {
                            previousTempValue = tempValue;

                            TemperatureSensorEvent.Builder event = new TemperatureSensorEvent.Builder(this);
                            event.temp(tempValue);

                            if (tempValue < minTempValue) {
                                minTempValue = tempValue;
                                event.maxTemp(maxTempValue);
                            }

                            if (tempValue > maxTempValue) {
                                maxTempValue = tempValue;
                                event.minTemp(minTempValue);
                            }

                            getEndpointContext().getEventService().fire(event.build());

                            // Send too hot and too cold events if needed.
                            if (tempValue < minThresholdValue) {
                                getEndpointContext().getEventService().fire( new TemperatureSensorTooColdEvent.Builder(this)
                                        .temp(tempValue)
                                        .minThreshold(minThresholdValue)
                                        .build());
                            } else if (maxThresholdValue < tempValue) {
                                getEndpointContext().getEventService().fire( new TemperatureSensorTooHotEvent.Builder(this)
                                        .temp(tempValue)
                                        .maxThreshold(maxThresholdValue)
                                        .build());                            }
                        }
                    }
                }
            }
        });

        measurementThread.start();
    }

    @Override
    public ReadOnlyDeviceAttribute<Date> startTimeProperty() {
        return startTime;
    }

    /**
     * Calculates a temperature value.
     *
     * @return a generated humidity value.
     */
    private float calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // round the temperature to 2 places and return
        int l = (int)((setPoint + delta) * 100.0);
        return(float)l / 100;
    }
    /**
     * Stops the device endpoint.
     * Unregisters from receiving sensor value updates.
     */
    @Override
    protected void stop() throws Exception {
        done.set(true);
        isOn = false;

        if (measurementThread != null) {
            synchronized(measurementThread) {
                measurementThread.notify();
            }
            measurementThread.join();
        }

        super.stop();
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> tempProperty() {
        return temp;
    }

    @Override
    public ReadOnlyDeviceAttribute<String> unitProperty() {
        return unit;
    }
}
