/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.type.temperature;

import oracle.iot.endpoint.AlertEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.AlertMessage;
import oracle.iot.message.AlertMessage.Severity;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a too hot condition on a temperature sensor
 * 
 * @param <T> the endpoint type that generated this event
 */
public class TemperatureSensorTooHotEvent<T extends TemperatureSensorEndpoint> extends AlertEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:temperature_sensor:too_hot";
    private static final String DESCRIPTION = "Temperature has reached the maximum temperature threshold";
    private Float temp = null;
    private Integer maxThreshold = null;
    private String unit = null;

    /**
     * Gets the temperature measurement contained in this event
     *
     * @return the temperature
     */
    public final Float getTemp() {
        return temp;
    }

    /**
     * Gets the max threshold of the device when this event was generated
     *
     * @return the max threshold
     */
    public Integer getMaxThreshold() {
        return maxThreshold;
    }

    /**
     * Measurement unit, such as Cel for Celsius.
     * for measurements in this this event
     *
     * @return the unit
     */
    public String getUnit() {
        return unit;
    }
    
    /**
     * Event type matching any {@code TemperatureSensorTooHotEvent}.
     */
    public static final EventType<TemperatureSensorTooHotEvent<? extends TemperatureSensorEndpoint>> ALERT =
            new EventType<>("TOO_HOT", AlertEvent.ANY);

    /**
     * Creates a new instance of {@code TemperatureSensorTooHotEven}.
     * 
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @param severity    the severity of the alert represented by this event
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected TemperatureSensorTooHotEvent(final T source,
                                          final EventType<? extends TemperatureSensorTooHotEvent> type,
                                          final Instant timestamp,
                                          final Severity severity)
    {
        super(source, type, FORMAT_URN, timestamp, DESCRIPTION, severity);
    }

    @Override
    protected AlertMessage.Builder constructAlertMessage(final AlertMessage.Builder builder) {
        AlertMessage.Builder out = builder;
        if(temp != null) builder.dataItem("temp", new Float(temp));
        if(maxThreshold != null) builder.dataItem("maxThreshold", new Float(maxThreshold));
        if(unit != null) builder.dataItem("unit", new String(unit));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(temp != null) out.append("temp",temp);
        if(maxThreshold != null) out.append("maxThreshold",maxThreshold);
        if(unit!= null) out.append("unit",unit);
        return out;
    }

    @Override
    public TemperatureSensorTooHotEvent<T> copy(final T newSource) {
        TemperatureSensorTooHotEvent event = new TemperatureSensorTooHotEvent(newSource, TemperatureSensorTooHotEvent.ALERT, 
                getTimestamp(), severity);
        event.temp = temp;
        event.maxThreshold = maxThreshold;
        event.unit = unit;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("TemperatureSensorTooHotEvent")).toString();
    }

    public static class Builder<T extends TemperatureSensorEndpoint> {
        T source;
        Instant timestamp;
        private Float temp = null;
        private Integer maxThreshold = null;
        private String unit = null;
        AlertMessage.Severity severity = Severity.SIGNIFICANT;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder temp(Float temp) { this.temp = temp; return this;}
        public Builder maxThreshold(Integer maxThreshold) { this.maxThreshold = maxThreshold; return this;}
        public Builder unit(String unit) { this.unit = unit; return this;}
        public Builder severity(Severity severity) { this.severity = severity; return this;}
        public TemperatureSensorTooHotEvent build() {
            TemperatureSensorTooHotEvent event = new TemperatureSensorTooHotEvent(source, TemperatureSensorTooHotEvent.ALERT, timestamp, severity);
            event.temp = temp;
            event.maxThreshold = maxThreshold;
            event.unit = unit;
            return event;
        }
    }
}
