/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.type.temperature;

import oracle.iot.endpoint.AlertEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.AlertMessage;
import oracle.iot.message.AlertMessage.Severity;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a too cold condition on a temperature sensor
 *
 * @param <T> the endpoint type that generated this event
 */
public class TemperatureSensorTooColdEvent<T extends TemperatureSensorEndpoint> extends AlertEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:temperature_sensor:too_cold";
    private static final String DESCRIPTION = "Temperature has reached the minimum temperature threshold";
    private Float temp = null;
    private Integer minThreshold = null;
    private String unit = null;

    /**
     * Gets the temperature measurement contained in this event
     *
     * @return the temperature
     */
    public final Float getTemp() {
        return temp;
    }

    /**
     * Gets the min threshold of the device when this event was generated
     *
     * @return the min threshold
     */
    public Integer getMinThreshold() {
        return minThreshold;
    }

    /**
     * Measurement unit, such as Cel for Celsius.
     * for measurements in this this event
     *
     * @return the unit
     */
    public String getUnit() {
        return unit;
    }
    
    /**
     * Event type matching any {@code TemperatureSensorTooColdEvent}.
     */
    public static final EventType<TemperatureSensorTooColdEvent<? extends TemperatureSensorEndpoint>> ALERT =
            new EventType<>("TOO_COLD", AlertEvent.ANY);

    /**
     * Creates a new instance of {@code TemperatureSensorTooColdEvent}.
     *
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @param severity    the severity of the alert represented by this event
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected TemperatureSensorTooColdEvent(final T source,
                                           final EventType<? extends TemperatureSensorTooColdEvent> type,
                                           final Instant timestamp,
                                           final Severity severity)
    {
        super(source, type, FORMAT_URN, timestamp, DESCRIPTION, severity);
    }

    @Override
    protected AlertMessage.Builder constructAlertMessage(final AlertMessage.Builder builder) {
        AlertMessage.Builder out = builder;
        if(temp != null) builder.dataItem("temp", new Float(temp));
        if(minThreshold != null) builder.dataItem("minThreshold", new Float(minThreshold));
        if(unit != null) builder.dataItem("unit", new String(unit));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(temp != null) out.append("temp",temp);
        if(minThreshold != null) out.append("minThreshold",minThreshold);
        if(unit!= null) out.append("unit",unit);
        return out;
    }

    @Override
    public TemperatureSensorTooColdEvent<T> copy(final T newSource) {
        TemperatureSensorTooColdEvent event = new TemperatureSensorTooColdEvent(newSource, TemperatureSensorTooColdEvent.ALERT,
                getTimestamp(), severity);
        event.temp = temp;
        event.minThreshold = minThreshold;
        event.unit = unit;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("TemperatureSensorTooColdEvent")).toString();
    }

    public static class Builder<T extends TemperatureSensorEndpoint> {
        T source;
        Instant timestamp;
        private Float temp = null;
        private Integer minThreshold = null;
        private String unit = null;
        AlertMessage.Severity severity = null; //Severity.SIGNIFICANT;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder temp(Float temp) { this.temp = temp; return this;}
        public Builder minThreshold(Integer minThreshold) { this.minThreshold = minThreshold; return this;}
        public Builder unit(String unit) { this.unit = unit; return this;}
        public Builder severity(Severity severity) { this.severity = severity; return this;}
        public TemperatureSensorTooColdEvent build() {
            TemperatureSensorTooColdEvent event = new TemperatureSensorTooColdEvent(source, TemperatureSensorTooColdEvent.ALERT, timestamp, severity);
            event.temp = temp;
            event.minThreshold = minThreshold;
            event.unit = unit;
            return event;
        }
    }
}
