/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.temperature;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;
import java.util.Date;

/**
 * Event representing a temperature sensor measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class TemperatureSensorEvent<T extends TemperatureSensorEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:temperature_sensor:attributes";
    private Float temp = null;
    private Integer maxThreshold = null;
    private Integer minThreshold = null;
    private Float minTemp= null;
    private Float maxTemp = null;
    private Date startTime = null;
    private String unit = null;

    /**
     * Gets the temperature measurement contained in this event
     *
     * @return the temperature
     */
    public final Float getTemp() {
        return temp;
    }

    /**
     * Gets the max threshold of the device when this event was generated
     *
     * @return the max threshold
     */
    public Integer getMaxThreshold() {
        return maxThreshold;
    }

    /**
     * Gets the min threshold of the device when this event was generated
     *
     * @return the min threshold
     */
    public Integer getMinThreshold() {
        return minThreshold;
    }

    /**
     * Gets the min temperature of the device when this event was generated
     *
     * @return the min temperature
     */
    public Float getMinTemp() {
        return minTemp;
    }

    /**
     * Gets the max temperature of the device when this event was generated
     *
     * @return the max temperature
     */
    public Float getMaxTemp() {
        return maxTemp;
    }

    /**
     * The time (measured in EPOCH) at which the system was powered ON or reset
     * when this event was generated
     *
     * @return the start time
     */
    public Date getStartTime() {
        return startTime;
    }
    
    /**
     * Measurement unit, such as Cel for Celsius.
     * for measurements in this this event
     *
     * @return the unit
     */
    public String getUnit() {
        return unit;
    }
    
    /**
     * Event type matching any {@code TemperatureSensorEvent}.
     */
    public static final EventType<TemperatureSensorEvent<? extends TemperatureSensorEndpoint>> REPORT =
            new EventType<>("TEMPERATURE", DataEvent.ANY);

    /**
     * Creates a new instance of {@code TemperatureSensorEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected TemperatureSensorEvent(final T source,
                                     final EventType<? extends TemperatureSensorEvent> type,
                                     final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(temp != null) builder.dataItem("temp", new Float(temp));
        if(maxThreshold != null) builder.dataItem("maxThreshold", new Float(maxThreshold));
        if(minThreshold != null) builder.dataItem("minThreshold", new Float(minThreshold)); 
        if(minTemp != null) builder.dataItem("minTemp", new Float(minTemp));
        if(maxTemp != null) builder.dataItem("maxTemp", new Float(maxTemp));
        if(startTime != null) builder.dataItem("startTime",new Double(startTime.getTime()));
        if(unit != null) builder.dataItem("unit", new String(unit));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(temp != null) out.append("temp",temp);
        if(maxThreshold != null) out.append("maxThreshold",maxThreshold);
        if(minThreshold != null) out.append("minThreshold",minThreshold);
        if(minTemp != null) out.append("minTemp",minTemp);
        if(maxTemp != null) out.append("maxTemp",maxTemp);
        if(startTime != null) out.append("startTime",startTime);
        if(unit!= null) out.append("unit",unit);
        return out;
    }

    @Override
    public TemperatureSensorEvent<T> copy(final T newSource) {
        TemperatureSensorEvent event = new TemperatureSensorEvent(newSource, TemperatureSensorEvent.REPORT, getTimestamp());
        event.temp = temp;
        event.maxThreshold = maxThreshold;
        event.minThreshold = minThreshold;
        event.minTemp= minTemp;
        event.maxTemp = maxTemp;
        event.startTime = startTime;
        event.unit = unit;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("TemperatureSensorEvent")).toString();
    }

    public static class Builder<T extends TemperatureSensorEndpoint> {
        T source;
        Instant timestamp;
        private Float temp = null;
        private Integer maxThreshold = null;
        private Integer minThreshold = null;
        private Float minTemp= null;
        private Float maxTemp = null;
        private Date startTime = null;
        private String unit = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder temp(Float temp) { this.temp = temp; return this;}
        public Builder maxThreshold(Integer maxThreshold) { this.maxThreshold = maxThreshold; return this;}
        public Builder minThreshold(Integer minThreshold) { this.maxThreshold = maxThreshold; return this;}
        public Builder minTemp(Float minTemp) { this.minTemp = minTemp; return this;}
        public Builder maxTemp(Float maxTemp) { this.maxTemp = maxTemp; return this;}
        public Builder startTime(Date startTime) { this.startTime = startTime; return this;}
        public Builder unit(String unit) { this.unit = unit; return this;}
        public TemperatureSensorEvent build() {
            TemperatureSensorEvent event = new TemperatureSensorEvent(source, TemperatureSensorEvent.REPORT, timestamp);
            event.temp = temp;
            event.maxThreshold = maxThreshold;
            event.minThreshold = minThreshold;
            event.minTemp = minTemp;
            event.maxTemp = maxTemp;
            event.startTime = startTime;
            event.unit = unit;
            return event;
        }
    }
}
