/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.app.motionactivatedcamera;

import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEndpoint;
import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEvent;
import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventHandlerReference;
import oracle.iot.event.EventService;
import oracle.iot.storage.StorageObject;
import oracle.iot.storage.StorageService;
import oracle.iot.storage.StorageSyncStatus;
import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.ResourceBundle;
import java.util.UUID;
import java.util.logging.Logger;

/**
 * Application that receives events from the MotionActivatedCameraAdapter that images and videos
 * have been uploaded to the storage service, and sends messages to the IoT CS.
 *
 * Note that this application is not the endpoint.  MotionActivatedCameraEndpointImpl is the endpoint.  This application
 * registers to receive events from the Gateway event subsystem, which the MotionActivatedCameraEndpointImpl sends over
 * the event subsystem, converts the event to a message, and sends the message to the IoT CS.
 *
 * There is also example code which downloads the images and videos from the storage service.
 * Typically the download might be used for something like downloading a firmware update for the
 * camera so it can be applied.  The example here is a bit contrived, but serves to show how
 * something can be downloaded from the storage service.
 */
@IoTDeviceApplication
public class MotionActivatedCameraApplication
    extends AbstractApplication implements MotionActivatedCameraEndpoint
{
    int numSecondsToRecord;
    private ResourceBundle messages;
    private ArrayList<EventHandlerReference> eventHandlers = new ArrayList<>();

    private EventService eventService;
    private Logger logger;
    private MessagingService messagingService;
    private SimpleReadOnlyDeviceAttribute<Date> imageTime;
    private SimpleReadOnlyDeviceAttribute<String> image;
    private StorageService storageService;

    // Property to specify the path to download files into (e.g. /tmp).
    private static final String DOWNLOAD_PATH_PROPERTY =
        "com.oracle.iot.sample.mac.storageDownloadPath";
    // Path to location to download files into (e.g. /tmp).
    private String downloadPath = null;


    /**
     * Creates a new instance of the {@code SensorApplication}
     */
    @Inject
    public MotionActivatedCameraApplication(MessagingService messagingService,
                                            EventService eventService,
                                            StorageService storageService,
                                            Logger logger)
    {
        this.messagingService = messagingService;
        this.eventService = eventService;
        this.storageService = storageService;
        this.logger = logger;
        this.messages = ResourceBundle.getBundle("Messages");
        this.downloadPath = System.getProperty(DOWNLOAD_PATH_PROPERTY, "/tmp");
    }

    /**
     * Downloads media from the storage service.
     *
     * @param downloadMediaUri the URI to the media to download.
     */
    private void downloadMedia(String downloadMediaUri) {
        try {
            StorageObject storageObject = storageService.createStorageObject(downloadMediaUri);

            downloadPath = downloadPath.endsWith(File.separator) ? downloadPath :
                downloadPath + File.separator;

            // Get the file name.
            int idx = downloadMediaUri.lastIndexOf('/');
            String fileName = downloadMediaUri.substring(idx + 1);

            // Write file using random file name and image or video extension.
            storageObject.setOutputStream(new FileOutputStream(downloadPath + UUID.randomUUID() +
                "-" + fileName));

            ObservableFuture<StorageSyncStatus> storageSyncStatusObservable =
                storageService.sync(storageObject);

            DownloadInvalidationListener imageDownloadInvalidationListener =
                new DownloadInvalidationListener();

            storageSyncStatusObservable.addListener(imageDownloadInvalidationListener);
        } catch(GeneralSecurityException | IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public ReadOnlyDeviceAttribute<String> imageProperty() {
        return image;
    }

    @Override
    public ReadOnlyDeviceAttribute<Date> imageTimeProperty() {
        return imageTime;
    }

    @Override
    public void record(int numSecondsToRecord) {
        this.numSecondsToRecord = numSecondsToRecord;
    }

    @Override
    protected void start() {
        final String eventFrom = messages.getString("eventfrom");

        // Now handle any events coming from the endpoint.
        synchronized(eventHandlers) {
            // Register to receive camera events from the endpoint implementation.
            eventHandlers.add(eventService.register(MotionActivatedCameraEvent.REPORT,
                (event) ->
                {
                    System.out.println("MotionActivatedCameraEvent = " + event.toDataMessage());

                    // Send message to IoT CS and media has been uploaded.
                    messagingService.submit(event.toDataMessage());

                    System.out.println(MessageFormat.format("Downloading |Date={0}|ID={1}|ImageName={2}|.",
                        new Date().toString(), getId() + eventFrom + event.getSource().getId(),
                        event.getImage()));

                    // Now download the file.
                    downloadMedia(event.getImage());
                }));
        }

        System.out.println("Motion activated camera application started.");
    }

    @Override
    protected void stop() {
        super.stop();

        // Unregister the event handlers.
        synchronized (eventHandlers) {
            for (EventHandlerReference handler : eventHandlers) {
                handler.unregister();
            }
        }
    }
}
