/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.motionactivatedcamera;

import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEndpoint;
import com.oracle.iot.sample.daf.type.motionactivatedcamera.MotionActivatedCameraEvent;
import oracle.iot.beans.InvalidationListener;
import oracle.iot.beans.Observable;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.event.EventService;
import oracle.iot.storage.StorageObject;
import oracle.iot.storage.StorageSyncStatus;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.logging.Logger;


/**
 * InvalidationListener that receives events when uploading an image completes.
 */
class ImageUploadInvalidationListener implements InvalidationListener {
    private Date imageTime = null;
    private EventService eventService = null;
    private Logger logger = null;
    private MotionActivatedCameraEndpoint motionActivatedCameraEndpoint;


    ImageUploadInvalidationListener(MotionActivatedCameraEndpoint motionActivatedCameraEndpoint,
                                    EventService eventService, Logger logger, Date imageTime)
    {
        this.motionActivatedCameraEndpoint= motionActivatedCameraEndpoint;
        this.eventService = eventService;
        this.logger = logger;
        this.imageTime = imageTime;
    }

    /**
     * When a image has completed uploading, get the data and send a MotionActivatedCameraEvent
     * over the event service.
     *
     * @inheritDoc
     */
    @Override
    public void invalidated(Observable observable) {
        System.out.println("Completed uploading image.");

        if ((eventService != null) && (observable instanceof ObservableFuture)) {
            ObservableFuture<StorageSyncStatus> observableFuture = (ObservableFuture) observable;

            try {
                StorageSyncStatus storageSyncStatus = observableFuture.get();

                if ((storageSyncStatus != null) &&
                    (storageSyncStatus.getState() == StorageSyncStatus.State.COMPLETED))
                {
                    StorageObject storageObject = storageSyncStatus.getStorageObject();

                    if (storageObject != null) {
                        URI uri = new URI(storageObject.getURI());
                        System.out.println("\nName: " + storageObject.getName());
                        System.out.println("\nDate: " + storageObject.getDate());
                        System.out.println("\nURI: " + storageObject.getURI());
                        System.out.println("\nLength: " + storageObject.getLength());
                        System.out.println("\nType: " + storageObject.getType());
                        System.out.println("\nEncoding: " + storageObject.getEncoding());

                        MotionActivatedCameraEvent.Builder eventBuilder =
                            new MotionActivatedCameraEvent.Builder(motionActivatedCameraEndpoint)
                                .imageTime(imageTime)
                                .image(uri.toString());

                        logger.info("Firing MotionActivatedCameraEvent for image upload.");
                        eventService.fire(eventBuilder.build());
                    }
                }
            } catch (ExecutionException | InterruptedException | URISyntaxException e) {
                e.printStackTrace();
            }
        }
    }
}
