/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.app.humiditysensor;

import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEvent;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEndpoint;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorTooHumidEvent;
import oracle.iot.app.AbstractApplication;
import oracle.iot.app.IoTDeviceApplication;

import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.message.AlertMessage;

import oracle.iot.messaging.MessagingService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * This humidity sensor application is an example of an application which is an endpoint itself and as such, generates
 * and sends humidity sensor readings to the IoT CS.
 *
 * Note: Some other sample applications receive events from their respective EndpointImpl's, then convert the event to a
 * message and send it to the IoT CS.  In these cases, the EndpointImpl is the endpoint and the application is sending
 * the message.  This sample *does not* do that.
 */
@IoTDeviceApplication
public class HumiditySensorApplication extends AbstractApplication implements HumiditySensorEndpoint {

    // Use random to generate probability that we'll exceed threshold.
    private static Random random = new Random();
    // The maximum threshold value, initialized to the default in the device model.
    private int maxThresholdValue = 80;
    // The attribute holding the maximum threshold value.
    private DeviceAttribute<Integer> maxThreshold;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;
    // the fluctuating humidity sensor value
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 5f;

    // The generated humidity value.
    private Integer humidityValue = 0;
    // The previous generated humidity value.
    private Integer previousHumidityValue = 0;
    // The attribute holding the humidity value.
    private ReadOnlyDeviceAttribute<Integer> humidity;

    private MessagingService ms;

    // thread used for generating messages
    private Thread measurementThread = null;
    private static final AtomicBoolean done = new AtomicBoolean(false);

    /**
     * Creates a new instance of the {@code SensorApplication}
     */
    @Inject
    public HumiditySensorApplication(MessagingService messagingService) {
        ms = messagingService;
    }

    @Override
    protected void start() {

        humidity = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "humidity",
                getEndpointContext().getEventService(),
                () -> humidityValue);

        maxThreshold = new SimpleDeviceAttribute<>(
                this,
                "maxThreshold",
                getEndpointContext().getEventService(),
                ()->maxThresholdValue,
                (s)-> {
                    System.out.println(new Date().toString() + " : " + getId() + " : " + " Set : \"maxThreshold\" = " + s);
                    maxThresholdValue=s;
                    ms.submit(new HumiditySensorEvent.Builder(this, Instant.now())
                            .humidity(humidityValue)
                            .maxThreshold(maxThresholdValue)
                            .build().toDataMessage());
                });

        // This thread simulates a humidity device and sends data and alert messages.
        measurementThread = new Thread(() -> {
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(2000);
                    } catch (Exception e) {
                    }
                }

                // Get a generated random number for the humidity value.
                humidityValue = calculateNextValue();
                System.out.println(new Date().toString() + " : " + getId() + " : Data : \"humidity\" = " + humidityValue);

                // Only send events if the humidity has changed
                if (humidityValue.intValue() != previousHumidityValue.intValue()) {
                    // Create and send an event indicating the new value.
                    ms.submit(new HumiditySensorEvent.Builder(this, Instant.now())
                                    .humidity(humidityValue)
                                    .build().toDataMessage());

                    // Send an alert if the humidity value is outside the threshold.
                    if (humidityValue > maxThresholdValue) {
                        System.out.println(new Date().toString() + " : " + getId() + " : Alert : \"humidity\" = " + humidityValue + ", \"maxThreshold\" = " +
                                maxThresholdValue);
                        ms.submit(new HumiditySensorTooHumidEvent.Builder(this, Instant.now())
                                        .humidity(humidityValue)
                                        .severity(AlertMessage.Severity.SIGNIFICANT)
                                        .build().toAlertMessage());
                    }
                }

            }
        });
        measurementThread.start();
    }


    @Override
    protected void stop() {
        done.set(true);

        if (measurementThread != null) {
            synchronized(measurementThread) {
                measurementThread.notify();
            }
            try {
                measurementThread.join();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        super.stop();
    }


    /**
     * Calculates a humidity value.
     *
     * @return a generated humidity value.
     */
    private int calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // return an rounded humidity point
        return (int)Math.round(setPoint + delta);
    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> humidityProperty() {
        return humidity;
    }

    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }
}
