/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.humidity;

import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEvent;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorEndpoint;
import com.oracle.iot.sample.daf.type.humidity.HumiditySensorTooHumidEvent;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.DeviceAttribute;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.message.AlertMessage;

import javax.inject.Inject;
import java.util.Date;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The humidity sensor endpoint represents the endpoint for a humidity sensor device.  The endpoint abstracts the
 * device within the IoT system.  Since the IoT system deals mainly with the endpoint, the underlying device can be
 * replaced without effecting the system.  The change in the IoT system is the association between the endpoint and
 * the new and old device.
 *
 * This implementation also acts as a humidity sensor by generating humidity values.  When used with a real device, this
 * class would communicate with the real device for getting and setting information.
 */
@IoTDeviceEndpoint
public class HumiditySensorEndpointImpl extends AbstractDeviceEndpoint implements HumiditySensorEndpoint {
    private static final AtomicBoolean done = new AtomicBoolean(false);
    private final static long OUTPUT_INTERVAL = 2000;
    // Use random to generate probability that we'll exceed threshold.
    private static Random random = new Random();
    // The maximum threshold value, initialized to the default in the device model.
    private int maxThresholdValue = 80;
    // The attribute holding the maximum threshold value.
    private DeviceAttribute<Integer> maxThreshold;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double angle = random.nextDouble() * 360f;
    // the fluctuating humidity sensor value
    private float setPoint = maxThresholdValue * .90f;
    // Used to generate a humidity value fluctuation around the 'set point'.
    private double amplitude = maxThresholdValue - setPoint + 5f;
    // The generated humidity value.
    private Integer humidityValue = 0;
    // The previous generated humidity value.
    private Integer previousHumidityValue = 0;
    // The attribute holding the humidity value.
    private ReadOnlyDeviceAttribute<Integer> humidity;
    private Thread measurementThread = null;

    @Inject
    public HumiditySensorEndpointImpl(Logger logger) {
        super();
        logger.log(Level.INFO, "Created new Humidity Sensor endpoint.");
    }

    /**
     * Calculates a humidity value.
     *
     * @return a generated humidity value.
     */
    private int calculateNextValue() {
        // calculate the delta based on the current angle
        final double delta = amplitude * Math.sin(Math.toRadians(angle));
        // rotate the angle
        angle += random.nextFloat() * 45f;
        // return an rounded humidity point
        return (int)Math.round(setPoint + delta);
    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> humidityProperty() {
        return humidity;
    }

    /**
     * Initializes the device endpoint.
     *
     * @param hardwareId the hardware ID of the device.
     */
    void init(String hardwareId) {

        humidity = new SimpleReadOnlyDeviceAttribute<>(
                this,
                "humidity",
                getEndpointContext().getEventService(),
                () -> humidityValue);

        maxThreshold = new SimpleDeviceAttribute<>(
                this,
                "maxThreshold",
                getEndpointContext().getEventService(),
                ()->maxThresholdValue,
                (s)-> {
                    System.out.println(new Date().toString() + " : " + getId() + " : " + " Set : \"maxThreshold\" = " + s);
                    maxThresholdValue=s;
                    getEndpointContext().getEventService().fire(
                            new HumiditySensorEvent.Builder(this)
                            .humidity(humidityValue)
                            .maxThreshold(maxThresholdValue)
                            .build());
                });
    }
    
    @Override
    public DeviceAttribute<Integer> maxThresholdProperty() {
        return maxThreshold;
    }

    @Override
    protected void start() throws Exception {
        done.set(false);
        super.start();

        // This thread simulates a humidity device and sends data and alert messages.
        measurementThread = new Thread(() -> {
            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(OUTPUT_INTERVAL);
                    } catch (Exception e) {
                    }
                }

                // Get a generated random number for the humidity value.
                humidityValue = calculateNextValue();
                System.out.println(new Date().toString() + " : " + getId() + " : Data : \"humidity\" = " + humidityValue);

                // Only send events if the humidity has changed
                if (humidityValue.intValue() != previousHumidityValue.intValue()) {
                    
                    // Create and send an event indicating the new value.
                    getEndpointContext().getEventService().fire(
                            new HumiditySensorEvent.Builder(this)
                                    .humidity(humidityValue)
                                    .build());

                    // Send an alert if the humidity value is outside the threshold.
                    if (humidityValue > maxThresholdValue) {
                        System.out.println(new Date().toString() + " : " + getId() + " : Alert : \"humidity\" = " 
                                + humidityValue + ", \"maxThreshold\" = " +  maxThresholdValue);
                        getEndpointContext().getEventService().fire(
                                new HumiditySensorTooHumidEvent.Builder(this)
                                        .humidity(humidityValue)
                                        .severity(AlertMessage.Severity.SIGNIFICANT)
                                        .build());
                    }
                }

            }
        });

        measurementThread.start();
    }

    @Override
    protected void stop() throws Exception {
        done.set(true);

        if (measurementThread != null) {
            synchronized(measurementThread) {
                measurementThread.notify();
            }
            try {
                measurementThread.join();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

        super.stop();
    }
}
