/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.adapter.humidity;

import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;

import javax.inject.Inject;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * The device adapter is the component that manages one or more devices within the IoT system.  It's job is to register
 * and un-register devices with the IoT system.
 */
@IoTDeviceAdapter
public class HumiditySensorDeviceAdapter extends AbstractDeviceAdapter {
    @Inject
    public HumiditySensorDeviceAdapter(Logger logger) {
        logger.log(Level.INFO, "Created new humidity sensor device adapter.");
    }

    /**
     * Starts the device adapter.
     */
    @Override
    protected void start() throws Exception {
        new Thread(() -> {
            synchronized(this) {
                try {
                    this.wait(5000);
                } catch(Exception e){};
            }

            // Get this device adapter's endpoint ID from the system.
            String endpointId = getEndpointContext().getEndpointId();

            // We might communicate with the device itself to get this information.
            String manufacturer = "Manufacturer:" + endpointId;
            String deviceClass = "HumiditySensorDevice";
            String serialNumber = "SerialNumber:" + endpointId;

            Metadata metadata = Metadata.builder().
                    protocol("device-model-sample").
                    protocolDeviceId(endpointId).
                    manufacturer(manufacturer).
                    deviceClass(deviceClass).
                    serialNumber(serialNumber).
                    build();

            // Create and register two devices.
            // Generate a unique and reproducible hardware ID for the device.
            String hardwareId1 = manufacturer + "-" + deviceClass + "-" + serialNumber + "-1";

            // Register a device with the IoT system.
            ObservableFuture<HumiditySensorEndpointImpl> future1 = registerDevice(
                    hardwareId1, metadata, HumiditySensorEndpointImpl.class,
                    (device) -> device.init(hardwareId1));

            // Listen for the new device representation being ready.
            future1.addListener((observable) -> {
                try {
                    HumiditySensorEndpointImpl device = future1.get();
                    getEndpointContext().getLogger().log(Level.FINE, "HumiditySensor 1 registered: " + device.getId());
                } catch (final Exception e) {
                    e.printStackTrace();
                }
            });

            // Generate a unique and reproducible hardware ID for the device.
            String hardwareId2 = manufacturer + "-" + deviceClass + "-" + serialNumber + "-2";

            // Register a device with the IoT system.
            ObservableFuture<HumiditySensorEndpointImpl> future2 = registerDevice(
                    hardwareId2, metadata, HumiditySensorEndpointImpl.class,
                    (device) -> device.init(hardwareId2));

            // Listen for the new device representation being ready.
            future2.addListener((observable) -> {
                try {
                    HumiditySensorEndpointImpl device = future2.get();
                    getEndpointContext().getLogger().log(Level.FINE, "HumiditySensor 2 registered: " + device.getId());
                } catch (final Exception e) {
                    e.printStackTrace();
                }
            });

        }).start();
    }

    /**
     * Stops the device adapter.
     */
    @Override
    protected void stop() throws Exception {
        super.stop();
    }
}
