/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.type.humidity;

import oracle.iot.endpoint.AlertEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.AlertMessage;
import oracle.iot.message.AlertMessage.Severity;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a too humid condition on a humidity sensor
 *
 * @param <T> the endpoint type that generated this event
 */
public class HumiditySensorTooHumidEvent<T extends HumiditySensorEndpoint> extends AlertEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:humidity_sensor:too_humid";
    private static final String DESCRIPTION = "Sample alert when humidity reaches the maximum humidity threshold";
    private Integer humidity = null;

    /**
     * Gets the humidity level that caused this event to be generated
     * 
     * @return the humidity
     */
    public final Integer getHumidity() {
        return humidity;
    }
    
    /**
     * Event type matching any {@code HumiditySensorTooHumidEvent}.
     */
    public static final EventType<HumiditySensorTooHumidEvent<? extends HumiditySensorEndpoint>> ALERT =
            new EventType<>("TOO_HUMID", AlertEvent.ANY);

    /**
     * Creates a new instance of {@code HumiditySensorTooHumidEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @param severity    the severity of the alert represented by this event
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected HumiditySensorTooHumidEvent(final T source,
                                      final EventType<? extends HumiditySensorTooHumidEvent> type,
                                      final Instant timestamp,
                                      final Severity severity)
    {
        super(source, type, FORMAT_URN, timestamp, DESCRIPTION, severity);
    }

    @Override
    protected AlertMessage.Builder constructAlertMessage(final AlertMessage.Builder builder) {
        AlertMessage.Builder out = builder;
        if(humidity != null) builder.dataItem("humidity", new Float(humidity));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(humidity != null) out.append("humidity",humidity);
        return out;
    }

    @Override
    public HumiditySensorTooHumidEvent<T> copy(final T newSource) {
        HumiditySensorTooHumidEvent event = new HumiditySensorTooHumidEvent(newSource, HumiditySensorTooHumidEvent.ALERT,  
                getTimestamp(), severity);
        event.humidity = humidity;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("HumiditySensorTooHumidEvent")).toString();
    }

    public static class Builder<T extends HumiditySensorEndpoint> {
        T source;
        Instant timestamp;
        Integer humidity;
        AlertMessage.Severity severity = Severity.SIGNIFICANT;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder humidity(Integer humidity) { this.humidity = humidity; return this;}
        public Builder severity(Severity severity) { this.severity = severity; return this;}
        public HumiditySensorTooHumidEvent build() {
            HumiditySensorTooHumidEvent event = new HumiditySensorTooHumidEvent(source, HumiditySensorTooHumidEvent.ALERT, timestamp, severity);
            event.humidity = humidity;
            return event;
        }
    }
}
