/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.helloworldadapter;

import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.DeviceEndpoint;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;

import javax.inject.Inject;
import java.util.concurrent.ExecutionException;
import java.util.logging.Logger;

/**
 * Sample hello world adapter that directly implements the HelloWorld device model.
 */
@IoTDeviceAdapter
public class HelloWorldAdapter extends AbstractDeviceAdapter {
    private HelloWorldEndpointImpl helloWorldEndpointImpl = null;
    private Logger logger;


    /**
     * Creates a new instance of the {@code SensorApplication}
     */
    @Inject
    public HelloWorldAdapter(Logger logger) {
        this.logger = logger;
    }

    @Override
    protected void start() {

        // Create and register a device.
        Metadata metadata = Metadata.builder()
            .deviceClass("HelloWorldAdapterDeviceClass")
            .manufacturer("HelloWorldAdapterManufacturer")
            .protocol("HelloWorldAdapterProtocol")
            .protocolDeviceId("HelloWorldAdapterProtocolAddress")
            .serialNumber("HelloWorldAdapterSerialNumber")
            .build();

        // Generate a globally unique and reproducible hardware ID for the device.
        String hardwareId = "HelloWorldAdapterHardwareId";
        logger.info("Registering Hello World device with hardwareId: " + hardwareId);

        ObservableFuture<HelloWorldEndpointImpl> future = registerDevice(hardwareId, metadata,
            HelloWorldEndpointImpl.class, (endpoint) -> endpoint.init(hardwareId));

        future.addListener((observable) -> {
            try {
                helloWorldEndpointImpl = future.get();
            } catch (final ExecutionException | InterruptedException e) {
                e.printStackTrace();
            }
        });
    }

    @Override
    protected void stop() {
        if (helloWorldEndpointImpl != null) {
            unregisterDevice(helloWorldEndpointImpl);
        }
    }
}
