/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.pulseoximeter;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a pulse oximeter measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class PulseOximeterEvent<T extends PulseOximeterEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:pulse_oximeter:attributes";
    private Float oxygenSaturation = null;
    private Integer heartRate = null;

    /**
     * Gets the oxygen saturation measurement contained in this event
     *
     * @return the oxygen saturation
     */
    public final Float getOxygenSaturation() {
        return oxygenSaturation;
    }
    
    /**
     * Gets the heart rate measurement contained in this event
     *
     * @return the heart rate
     */
    public Integer getHeartRate() {
        return heartRate;
    }

    /**
     * Event type matching any {@code PulseOximeterEvent}.
     */
    public static final EventType<PulseOximeterEvent<? extends PulseOximeterEndpoint>> REPORT =
            new EventType<>("PULSE_OXIMETER", DataEvent.ANY);

    /**
     * Creates a new instance of {@code PulseOximeterEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected PulseOximeterEvent(final T source,
                                        final EventType<? extends PulseOximeterEvent> type,
                                        final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(oxygenSaturation != null) builder.dataItem("oxygenSaturation", new Float(oxygenSaturation));
        if(heartRate != null) builder.dataItem("heartRate", new Float(heartRate));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(oxygenSaturation != null) out.append("oxygenSaturation",oxygenSaturation);
        if(heartRate != null) out.append("heartRate",heartRate);
        return out;
    }

    @Override
    public PulseOximeterEvent<T> copy(final T newSource) {
        PulseOximeterEvent event = new PulseOximeterEvent(newSource, PulseOximeterEvent.REPORT, getTimestamp());
        event.oxygenSaturation = oxygenSaturation ;
        event.heartRate = heartRate;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("PulseOximeterEvent")).toString();
    }

    public static class Builder<T extends PulseOximeterEndpoint> {
        T source;
        Instant timestamp;
        private Float oxygenSaturation = null;
        private Integer heartRate = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder oxygenSaturation (Float oxygenSaturation ) { this.oxygenSaturation = oxygenSaturation ; return this;}
        public Builder heartRate(Integer heartRate) { this.heartRate = heartRate; return this;}
        public PulseOximeterEvent build() {
            PulseOximeterEvent event = new PulseOximeterEvent(source, PulseOximeterEvent.REPORT, timestamp);
            event.oxygenSaturation  = oxygenSaturation ;
            event.heartRate = heartRate;
            return event;
        }
    }
}
