/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.health;

import jdk.phd.Phd;
import jdk.phd.report.ConfigReport;
import jdk.phd.report.NumericMeasurement;
import jdk.phd.report.ScanReport;
import jdk.phd.report.spec.WeighingScale;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.bodyweightscale.BodyWeightScaleEvent;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.logging.Level;

/**
 * Implementation of the IEEE 11073 weighing scale IoT Endpoint.
 */
@IoTDeviceEndpoint
public class WeightScaleEndpoint extends PersonalHealthEndpoint implements com.oracle.iot.sample.daf.type.bodyweightscale.BodyWeightScaleEndpoint {

    private final SimpleReadOnlyDeviceAttribute<Float> bodyWeight;
    private EventService es;

    @Inject
    public WeightScaleEndpoint(EventService es)
    {
        super();
        this.es = es;
        bodyWeight = new SimpleReadOnlyDeviceAttribute<>(this, "bodyWeight", es);
    }


    @Override
    public void scanReported(Phd phd, ScanReport scanReport) {
        logger.log(Level.FINE, "PhdHandler.scanReported");

        WeighingScale wsScanReport = (WeighingScale)scanReport;

        if(wsScanReport.getBodyWeight() == null || wsScanReport.getBodyWeight().size() == 0)
            return;

        // Dispatch each weight measurement received.
        for(NumericMeasurement nm: wsScanReport.getBodyWeight()) {
            dispatch(nm.getTimestamp(), (double)nm.getValue());
        }
    }

    @Override
    public boolean configReported(Phd phd, ConfigReport configReport) {
        logger.log(Level.FINE, "PhdHandler.configReported");
        if(configReport.getConfig().getConfigId() == 0x5DC) {
            logger.log(Level.INFO, "accepted configuration from weight scale device: " + address);
            return true;
        }
        logger.log(Level.FINE, "Config " + configReport.getConfig().getConfigId() + " reported - rejecting.");
        return false;
    }

    /**
     * Dispatch an extended weighing scale measurement
     *
     * @param time the timestamp delivered with this measurement
     * @param bodyWeight the body weight received in this measurement
     */
    void dispatch(long time, double bodyWeight) {
        logger.log(Level.FINE, "generating event: " + bodyWeight + " from device: " + address);
        this.bodyWeight.notifyValueUpdated((float)bodyWeight);

        es.fire(new BodyWeightScaleEvent.Builder(this,Instant.ofEpochMilli(time))
                .bodyWeight((float)bodyWeight)
                .build());
    }

    @Override
    public ReadOnlyDeviceAttribute<Float> bodyWeightProperty() {
        return bodyWeight;
    }

    @Override
    public Float getBodyWeight() {
        return bodyWeight.getValue();
    }

    @Override
    public String getHardwareId() {
        return "WeightScalEndpoint-" + System.currentTimeMillis();
    }
}
