/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.health;

import jdk.phd.Phd;
import jdk.phd.report.ConfigReport;
import jdk.phd.report.NumericMeasurement;
import jdk.phd.report.ScanReport;
import oracle.iot.beans.property.IntegerProperty;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.bloodpressuremonitor.BloodPressureMonitorEvent;
import com.oracle.iot.sample.daf.type.heartratemonitor.HeartRateMonitorEndpoint;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.HashMap;
import java.util.logging.Level;

@IoTDeviceEndpoint
public class BloodPressureMonitorEndpoint extends PersonalHealthEndpoint implements com.oracle.iot.sample.daf.type.bloodpressuremonitor.BloodPressureMonitorEndpoint, HeartRateMonitorEndpoint {

    private HashMap<Long, NumericMeasurement> systolicMeasurements = new HashMap<>();
    private HashMap<Long, NumericMeasurement> diastolicMeasurements = new HashMap<>();
    private HashMap<Long, NumericMeasurement> heartRateMeasurements = new HashMap<>();

    private final SimpleReadOnlyDeviceAttribute<Integer> heartRate;
    private final SimpleReadOnlyDeviceAttribute<Integer> systolic;
    private final SimpleReadOnlyDeviceAttribute<Integer> diastolic;
    private IntegerProperty heartRateProp;
    private IntegerProperty systolicProp;
    private IntegerProperty diastolicProp;
    private EventService es;

    @Inject
    public BloodPressureMonitorEndpoint(EventService es) {
        super();
        this.es = es;
        heartRate = new SimpleReadOnlyDeviceAttribute<>(this, "heartRate", es);
        systolic = new SimpleReadOnlyDeviceAttribute<>(this, "systolic", es);
        diastolic = new SimpleReadOnlyDeviceAttribute<>(this, "diastolic", es);
    }

    @Override
    public void scanReported(Phd phd, ScanReport scanReport) {
        logger.log(Level.FINE, "PhdHandler.scanReported");

        jdk.phd.report.spec.BloodPressureMonitor bpmScanReport = (jdk.phd.report.spec.BloodPressureMonitor)scanReport;

        // Add all measurement components
        for(NumericMeasurement nm: bpmScanReport.getDiastolicPressure())
            diastolicMeasurements.put(nm.getTimestamp(), nm);
        for(NumericMeasurement nm: bpmScanReport.getSystolicPressure())
            systolicMeasurements.put(nm.getTimestamp(), nm);
        for(NumericMeasurement nm: bpmScanReport.getPulseRate())
            heartRateMeasurements.put(nm.getTimestamp(), nm);

        reportMeasurements();
    }

    @Override
    public boolean configReported(Phd phd, ConfigReport configReport) {
        logger.log(Level.FINE, "PhdHandler.configReported");
        if(configReport.getConfig().getConfigId() == 0x2BC) {
            logger.log(Level.INFO, "accepted configuration from blood pressure monitor: " + address);
            return true;
        }
        logger.log(Level.FINE, "Config " + configReport.getConfig().getConfigId() + " reported - rejecting.");
        return false;
    }

    /**
     * Search stored individual measurement values and attempt to assemble consolidated events for dispatch
     */
    void reportMeasurements()
    {
        for(Long time : heartRateMeasurements.keySet()) dispatchEvent(time);
        for(Long time : diastolicMeasurements.keySet()) dispatchEvent(time);
        for(Long time : systolicMeasurements.keySet()) dispatchEvent(time);
    }

    /**
     * Consolidate all measurement events for a given event time and dispatch
     *
     * @param time the time stamp of the measurements to find
     */
    void dispatchEvent(Long time)
    {
        int heartRateVal = -1;
        int diastolicVal = -1;
        int systolicVal = -1;
        NumericMeasurement hr = heartRateMeasurements.remove(time);
        NumericMeasurement dbp = diastolicMeasurements.remove(time);
        NumericMeasurement sbp = systolicMeasurements.remove(time);

        if(hr != null) heartRateVal = (int)hr.getValue();
        if(dbp != null) diastolicVal = (int)dbp.getValue();
        if(sbp != null) systolicVal = (int)sbp.getValue();

        dispatch(time, heartRateVal, diastolicVal, systolicVal);
    }

    void dispatch(long time, int heartRateVal, int diastolicVal, int systolicVal) {
        logger.log(Level.FINE, "generating event: " + heartRateVal + ":" + diastolicVal + ":"
                + systolicVal + " from device: " + address);

        // Update the values if signaled, and grab a valid update time.
        if(systolicVal != -1)  this.systolic.notifyValueUpdated(systolicVal);
        if(diastolicVal != -1) this.diastolic.notifyValueUpdated(diastolicVal);
        if(heartRateVal != -1) this.heartRate.notifyValueUpdated(heartRateVal);

        es.fire(new BloodPressureMonitorEvent.Builder(this,Instant.ofEpochMilli(time))
                .diastolicPressure(diastolicVal)
                .heartRate(heartRateVal)
                .systolicPressure(systolicVal)
                .build());

    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> heartRateProperty() {
        return heartRate;
    };

    @Override
    public Integer getHeartRate() {
        return heartRate.getValue();
    };

    @Override
    public ReadOnlyDeviceAttribute<Integer> systolicPressureProperty() {
        return systolic;
    }

    @Override
    public Integer getSystolicPressure() {
        return systolic.getValue();
    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> diastolicPressureProperty() {
        return diastolic;
    }

    @Override
    public Integer getDiastolicPressure() {
        return diastolic.getValue();
    }
}
