/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.heartratemonitor;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a heart rate measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class HeartRateMonitorEvent<T extends HeartRateMonitorEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:heart_rate_monitor:attributes";
    private Integer heartRate = null;

    /**
     * Gets the heart rate measurement contained in this event
     *
     * @return the heart rate
     */
    public final Integer  getHeartRate() {
        return heartRate;
    }

    /**
     * Event type matching any {@code HeartRateMonitorEvent}.
     */
    public static final EventType<HeartRateMonitorEvent<? extends HeartRateMonitorEndpoint>> REPORT =
            new EventType<>("HEART_RATE", DataEvent.ANY);

    /**
     * Creates a new instance of {@code HeartRateMonitorEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected HeartRateMonitorEvent(final T source,
                                   final EventType<? extends HeartRateMonitorEvent> type,
                                   final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(heartRate != null) builder.dataItem("heartRate", new Float(heartRate));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(heartRate != null) out.append("heartRate",heartRate);
        return out;
    }

    @Override
    public HeartRateMonitorEvent<T> copy(final T newSource) {
        HeartRateMonitorEvent event = new HeartRateMonitorEvent(newSource, HeartRateMonitorEvent.REPORT, getTimestamp());
        event.heartRate = heartRate ;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("HeartRateMonitorEvent")).toString();
    }

    public static class Builder<T extends HeartRateMonitorEndpoint> {
        T source;
        Instant timestamp;
        private Integer  heartRate = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder heartRate (Integer  heartRate ) { this.heartRate = heartRate ; return this;}
        public HeartRateMonitorEvent build() {
            HeartRateMonitorEvent event = new HeartRateMonitorEvent(source, HeartRateMonitorEvent.REPORT, timestamp);
            event.heartRate  = heartRate ;
            return event;
        }
    }
}
