/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.type.bodyweightscale;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

/**
 * Event representing a body weight measurement
 *
 * @param <T> the endpoint type that generated this event
 */
public class BodyWeightScaleEvent<T extends BodyWeightScaleEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:body_weight_scale:attributes";
    private Float bodyWeight = null;
    
    /**
     * Gets the body weight measurement contained in this event
     *
     * @return the body weight
     */
    public final Float getBodyWeight() {
        return bodyWeight;
    }
    
    /**
     * Event type matching any {@code  BodyWeightScaleEvent}.
     */
    public static final EventType<BodyWeightScaleEvent<? extends BodyWeightScaleEndpoint>> REPORT =
            new EventType<>("BODY_WEIGHT", DataEvent.ANY);

    /**
     * Creates a new instance of {@code  BodyWeightScaleEvent}.
     *
     * @param source      the event source.
     * @param type        the event type.
     * @param timestamp   the time when the event occurred.
     * @throws NullPointerException if the source or timestamp parameter is {@code null}.
     */
    protected BodyWeightScaleEvent(final T source,
                                        final EventType<? extends BodyWeightScaleEvent> type,
                                        final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        if(bodyWeight != null) builder.dataItem("bodyWeight", new Float(bodyWeight));
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);
        if(bodyWeight != null) out.append("bodyWeight",bodyWeight);
        return out;
    }

    @Override
    public BodyWeightScaleEvent<T> copy(final T newSource) {
        BodyWeightScaleEvent event = new BodyWeightScaleEvent(newSource, BodyWeightScaleEvent.REPORT, getTimestamp());
        event.bodyWeight = bodyWeight ;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("BodyWeightScaleEvent")).toString();
    }

    public static class Builder<T extends BodyWeightScaleEndpoint> {
        T source;
        Instant timestamp;
        private Float bodyWeight = null;
        public Builder(final T source) {  this.source = source; this.timestamp = Instant.now();}
        public Builder(final T source,final Instant timestamp) {  this.source = source; this.timestamp = timestamp;}
        public Builder bodyWeight (Float bodyWeight ) { this.bodyWeight = bodyWeight ; return this;}
        public BodyWeightScaleEvent build() {
            BodyWeightScaleEvent event = new BodyWeightScaleEvent(source, BodyWeightScaleEvent.REPORT, timestamp);
            event.bodyWeight  = bodyWeight ;
            return event;
        }
    }
}
