/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.deviceio.uart;

import jdk.dio.*;
import jdk.dio.uart.UART;
import jdk.dio.uart.UARTConfig;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;

import javax.inject.Inject;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Device adapter for UART based simulated thermometer devices.
 */
@IoTDeviceAdapter
public class UartThermometerAdapter extends AbstractDeviceAdapter {

    /**
     * Name of serial device to open
     */
    private static final String UART_DEVICE_NAME = "ttyAMA0";

    /**
     * Decimal USB Manufacturer of Prolific
     * for PL-2303HX USB to serial chip
     */
    private static final String UART_DEVICE_MANUFACTURER = "1659";

    /**
     * Local logger instance
     */
    private Logger logger;

    /**
     * UART device
     */
    private UART uart;

    /**
     * Constructs an instance using dependency injection.
     * @param logger logger associated with this device adapter
     */
    @Inject
    public UartThermometerAdapter(Logger logger) {
        this.logger = logger;
        logger.log(Level.INFO, "Created new UART thermometer adapter.");
    }

    /**
     * Starts the device adapter.
     *
     * This method will initiate connection to the UART device. If successful,
     * it will register a UART thermometer endpoint.
     */
    @Override
    protected void start() throws Exception {
        openUart();
    }

    /**
     * Stops the device adapter.
     * Calls the {@code super.stop()} method which takes care of stopping
     * and unregistering all devices created by this adapter. Unregisters
     * the device discovery listener.
     */
    @Override
    protected void stop() throws Exception {
        super.stop();
        uart.close();
    }

    /**
     * Opens the UART device, and if successful creates a uart thermometer endpoint
     */
    private void openUart() {
        String uartDevice = System.getProperty("com.oracle.iot.sample.uart", UART_DEVICE_NAME);
        logger.log(Level.FINE, "Opening UART device: {0}", uartDevice);
        UARTConfig config = new UARTConfig(uartDevice,
                DeviceConfig.DEFAULT,
                115200,
                UARTConfig.DATABITS_8,
                UARTConfig.PARITY_NONE,
                UARTConfig.STOPBITS_1,
                UARTConfig.FLOWCONTROL_NONE);
        try {
            uart = DeviceManager.open(config);
            logger.log(Level.FINEST, "UART device opened. Registering endpoint.");
            registerUartDevice(uartDevice);
        } catch (Exception e) {
            // There are 5 exception types that can be thrown by the open method.
            // For this adapter, simply display the exception and return
            logger.warning("Exception while opening UART device: " + e + " " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void registerUartDevice(String uartDevice) {
        Metadata metadata = Metadata.builder().
                protocol("uart-sample").
                protocolDeviceId(uartDevice).
                manufacturer(UART_DEVICE_MANUFACTURER).
                deviceClass("UartThermometerDevice").
                serialNumber(getSerialNumber(uartDevice)).
                build();

        String hardwareId = "UartThermometer-" + getSerialNumber(uartDevice);

        ObservableFuture<UartThermometerEndpoint> future = registerDevice(
                hardwareId, metadata, UartThermometerEndpoint.class,
                (device) -> initDevice( hardwareId, device, uart));
    }

    /**
     * Initialize newly created thermometer endpoint
     *
     * @param hardwareId a globally unqiue hardware ID.
     * @param device     the uart thermometer device to initialize
     * @param uart       the uart stream to use for communication with the device
     */
    private void initDevice(String hardwareId, UartThermometerEndpoint device, UART uart) {

            logger.log(Level.FINEST, "initializing UART endpoint");
            try {
                uart.setReceiveTimeout(5000);
            } catch (IOException e) {
                logger.log(Level.WARNING, "exception setting UART receive timeout.");
                e.printStackTrace();
                return;
            }
            InputStream serialInputStream = Channels.newInputStream(uart);
            OutputStream serialOutputStream = Channels.newOutputStream(uart);
            device.init(hardwareId, serialInputStream, serialOutputStream);
    }

    /**
     * Return the serial number for a Uart Thermometer Endpoint
     *
     * @return the generated serial number
     */
    private String getSerialNumber(String uartDevice) {
        // Here we use the context endpoint ID has a basis to compute our 
        // serial number, since the device doesn't provide one
        // Having a real serial number provided by the device is preferred,
        // but if not available, a pseudo-serial number must be carefully chosen
        // to be unique per adapter (and the adapter endpoint ID can help 
        // providing global uniquess)
        return getEndpointContext().getEndpointId() + ":" + uartDevice;
    }
}
