/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.adapter.deviceio.rs485;

import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.AbstractPollingDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEndpoint;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEvent;
import oracle.iot.messaging.IoTResource;

import jdk.dio.uart.UART;

import javax.inject.Inject;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.Override;
import java.lang.Short;
import java.time.Instant;
import java.time.Duration;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.nio.ByteBuffer;


/**
 * RS485 Thermometer Endpoint class.
 */
@IoTDeviceEndpoint
public class Rs485ThermometerEndpoint extends AbstractPollingDeviceEndpoint implements ThermometerEndpoint {

    /** localized output messages */
    private ResourceBundle messages;
    private SimpleReadOnlyDeviceAttribute<Float> temperature;
    private Logger logger;
    private String hardwareId;
    private UART rs485;


    private static final int MAX_CMD_SIZE = 8;
    private static final int MAX_RES_SIZE = 7;
    private static final byte SENSOR_ADDR = 0x31;

    /**
     * @param logger logger associated with this endpoint
     */
    @Inject
    public Rs485ThermometerEndpoint(Logger logger) {
        this.logger = logger;
        messages = ResourceBundle.getBundle("Messages");
        logger.log(Level.INFO, "Created new RS-485 thermometer endpoint.");
    }

    /**
     * Initialize this endpoint with a serial port input stream
     *
     * @param hardwareId        a globally unique hardware ID.
     * @param serialInputStream Physical address of the device on the sample protocol.
     */
    void init(String hardwareId, UART uart) {
        this.hardwareId = hardwareId;
        rs485 = uart;

        /* Set the polling rate to 5 seconds. Every 5 seconds, the poll method will be called to read the data from the buffer */
        setPollingRate(Duration.ofMillis(5000));

        temperature = new SimpleReadOnlyDeviceAttribute<>(
                this, "temperature", getEndpointContext().getEventService());
        logger.log(Level.INFO, "Starting RS-485 thermometer listener...");
    }

    /**
     * Processes a reading from the UART thermometer and sends the measurement message
     * to the server.
     *
     * @param t the temperature data read from the UART thermometer
     */
    private void processTemperatureReading(Float t) {
        try {
            System.out.printf(messages.getString("temperature"), getEndpointContext().getEndpointId(),t);

            getEndpointContext().getMessagingService().submit(new ThermometerEvent.Builder(this)
                    .temperature(t)
                    .build().toDataMessage());
            temperature.notifyValueUpdated(t);
        } catch(Exception e) {
            logger.log(Level.FINEST, "Invalid temperature value: {0}", t);
        }
    }

    /**
     * Method that is invoked to do the actual polling.
     *
     * The poll rate has been set to once every 5 seconds, which means the poll() will be called every 5 seconds
     */
    @Override
    protected void poll() {

        boolean res = false;

        logger.log(Level.INFO, "======= Reading Papouch TM digital temperature sensor with address 0x31 ========");
        res = TemperatureDeviceRTU.binWriteUARTRS485(TemperatureDeviceRTU.getTempReadCmd(SENSOR_ADDR), this.rs485, this.logger);
        if (res == true) {
            double temp = TemperatureDeviceRTU.readUARTRS485(SENSOR_ADDR, this.rs485, this.logger);
            processTemperatureReading((float)temp);
        }
    }

    /**
     * Device framework resource access method.
     *
     * @return the temperature property for this endpoint
     */
    @IoTResource
    public ReadOnlyDeviceAttribute<Float> temperatureProperty() {
        return temperature;
    }

    @Override
    public Float getTemperature() {
        return temperature.getValue();
    }

    private static class TemperatureDeviceRTU
    {


        private static byte[] getTempReadCmd(byte sensorAddr) {

            switch(sensorAddr)
            {
                case SENSOR_ADDR: {
                    return new byte[] {(byte)0x31, (byte)0x04, (byte)0x00, (byte)0x01, (byte)0x00, (byte)0x01, (byte)0x65, (byte)0xfa};
                }
                default: break;
            }
            return new byte[] {(byte)0x32, (byte)0x04, (byte)0x00, (byte)0x01, (byte)0x00, (byte)0x01, (byte)0x65, (byte)0xc9};
        }

        /*
        parses the incoming response from TQS3 sensor

        Output: <address><1 byte><read type><1 byte><data length><2 bytes><temperature><2 bytes><crc><2 bytes>
        E.g.: 0x32, 0x04, 0x02, 0x01, 0x28, 0xbc, 0xba
        */
        private static double readUARTRS485(byte sensorAddr, UART rs485, Logger logger) {


            try{
                logger.log(Level.INFO, "Reading from RS-485");

                ByteBuffer resBuf = ByteBuffer.allocateDirect(MAX_RES_SIZE);

                int bufferLength = 0;

                bufferLength = rs485.read(resBuf);

                logger.log(Level.INFO, "Received: " + bufferLength + " [" + toHexa(resBuf) + "]");

                /* parse incoming response */
                while (resBuf.get() != sensorAddr);
                resBuf.position(resBuf.position() + 2);


                short val = Short.MAX_VALUE;
                val = resBuf.getShort();
                logger.log(Level.FINE, "Temperature: " + (float) (val / 10f) + " C");

                return (double)(val / 10f);

            } catch (Exception e) {
                logger.log(Level.WARNING, "RT: Caught exception: " + e);
                return (double) Short.MAX_VALUE;
            }

        } //end read

        private static boolean binWriteUARTRS485(byte[] cmdReadTemp, UART rs485, Logger logger) {

            try {
                ByteBuffer cmdBuf = ByteBuffer.allocateDirect(MAX_CMD_SIZE);
                int n = rs485.write((ByteBuffer) ((ByteBuffer) cmdBuf.clear()).put(cmdReadTemp).flip());
                logger.log(Level.FINE, "Sent: " + n + " [" + toHexa(ByteBuffer.wrap(cmdReadTemp)) + "]");
            } catch (Exception e) {
                logger.log(Level.WARNING, "WT: Caught exception: " + e);
                return false;
            }

            return true;
        } //end write


        private static String toHexa(ByteBuffer buffer) {
            buffer.flip();
            StringBuilder b = new StringBuilder();
            for (int i = 0; i < buffer.remaining(); i++) {
                b.append(Integer.toHexString(buffer.get(i) & 0xFF)).append(' ');
            }
            return b.toString();
        }

    }

}
