/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.deviceio.rs485;

import jdk.dio.DeviceConfig;
import jdk.dio.DeviceManager;
import jdk.dio.uart.UART;
import jdk.dio.uart.UARTConfig;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;

import javax.inject.Inject;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Device adapter for RS485 based thermometer devices.
 */
@IoTDeviceAdapter
public class Rs485ThermometerAdapter extends AbstractDeviceAdapter {

    /**
     * Name of serial device to open
     */
    private static final String UART_DEVICE_NAME = "ttymxc4";

    /**
     * Decimal USB Manufacturer of Prolific
     * for PL-2303HX USB to serial chip
     */
    private static final String UART_DEVICE_MANUFACTURER = "32";

    /**
     * Local logger instance
     */
    private Logger logger;

    /**
     * UART device
     */
    private UART uart;

    /**
     * Constructs an instance using dependency injection.
     * @param logger logger associated with this device adapter
     */
    @Inject
    public Rs485ThermometerAdapter(Logger logger) {
        this.logger = logger;
        logger.log(Level.INFO, "Created new RS-485 thermometer adapter.");
    }

    /**
     * Starts the device adapter.
     *
     * This method will initiate connection to the UART device. If successful,
     * it will register a UART thermometer endpoint.
     */
    @Override
    protected void start() throws Exception {
        openUart();
    }

    /**
     * Stops the device adapter.
     * Calls the {@code super.stop()} method which takes care of stopping
     * and unregistering all devices created by this adapter. Unregisters
     * the device discovery listener.
     */
    @Override
    protected void stop() throws Exception {
        super.stop();
        uart.close();
    }

    /**
     * Opens the UART device, and if successful creates a rs485 thermometer endpoint
     */
    private void openUart() {

        String uartDevice = System.getProperty("com.oracle.iot.sample.uart", UART_DEVICE_NAME);

        logger.log(Level.FINE, "Opening RS-485 device: {0}", uartDevice);
        UARTConfig config = new UARTConfig(uartDevice,
                DeviceConfig.DEFAULT,
                9600,
                UARTConfig.DATABITS_8,
                UARTConfig.PARITY_NONE,
                UARTConfig.STOPBITS_1,
                UARTConfig.FLOWCONTROL_NONE);
        try {
            uart = DeviceManager.open(config);
            logger.log(Level.FINEST, "RS-485 device opened. Registering endpoint.");
            registerUartDevice();
        } catch (Exception e) {
            // There are 5 exception types that can be thrown by the open method.
            // For this adapter, simply display the exception and return
            logger.warning("Exception while opening RS-485 device: " + e + " " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void registerUartDevice() {
        Metadata metadata = Metadata.builder().
                protocol("Proprietary Command over RS-485").
                protocolDeviceId(UART_DEVICE_NAME).
                manufacturer(UART_DEVICE_MANUFACTURER).
                deviceClass("Thermometer").
                serialNumber(getSerialNumber()).
                build();

        String hardwareId = "Rs485Thermometer-" + getSerialNumber();

        ObservableFuture<Rs485ThermometerEndpoint> future = registerDevice(
                hardwareId, metadata, Rs485ThermometerEndpoint.class,
                (device) -> initDevice(hardwareId, device, uart));
    }

    /**
     * Initialize newly created thermometer endpoint
     *
     * @param hardwareId a globally unique hardware ID.
     * @param device     the rs485 thermometer device to initialize
     * @param uart       the rs485 stream to use for communication with the device
     */
    private void initDevice(String hardwareId, Rs485ThermometerEndpoint device, UART uart) {
        logger.log(Level.FINEST, "initializing RS-485 endpoint");

        try {
            uart.setReceiveTimeout(5000);
        } catch (IOException e) {
            logger.log(Level.WARNING, "exception setting RS-485 receive timeout.");
            e.printStackTrace();
            return;
        }

        device.init(hardwareId, uart);
    }

    /**
     * Return the serial number for a Uart Thermometer Endpoint
     *
     * @return the generated serial number
     */
    private String getSerialNumber() {
        // Here we use the context endpoint ID has a basis to compute our 
        // serial number, since the device doesn't provide one
        // Having a real serial number provided by the device is preferred,
        // but if not available, a pseudo-serial number must be carefully chosen
        // to be unique per adapter (and the adapter endpoint ID can help 
        // providing global uniquess)
        return getEndpointContext().getEndpointId() + ":" + UART_DEVICE_NAME;
    }
}
