/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.adapter.deviceio.rs232;

import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.AbstractPollingDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEndpoint;
import com.oracle.iot.sample.daf.type.thermometer.ThermometerEvent;
import oracle.iot.messaging.IoTResource;
import jdk.dio.uart.UART;

import javax.inject.Inject;
import java.lang.Override;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.Duration;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.io.IOException;

/**
 * RS232 Thermometer Endpoint class.
 */
@IoTDeviceEndpoint
public class Rs232ThermometerEndpoint extends AbstractPollingDeviceEndpoint implements ThermometerEndpoint {
    // offset in byte array/ascii characters returned by temperature sensor
    public static final byte SENSOR_OUTPUT_BYTES_NUMBER     = 8;
    public static final byte SENSOR_OUTPUT_SIGN             = 0;
    public static final byte SENSOR_OUTPUT_TEMP_VALUE       = 1;
    public static final byte SENSOR_OUTPUT_DECIMAL_POINT    = 4;
    public static final byte SENSOR_OUTPUT_TEENTH_TEMP_VALUE= 5;
    public static final byte SENSOR_OUTPUT_CELSIUS          = 6;
    public static final byte SENSOR_OUTPUT_ENTER            = 7;
    // ASCII codes for characters returned by temperature sensor
    public static final byte ASCII_CODE_SIGN 	     = 0x2B;
    public static final byte ASCII_CODE_DECIMAL_POINT= 0x2E;
    public static final byte ASCII_CODE_CELSIUS      = 0x43;
    public static final byte ASCII_CODE_ZERO         = 0x30;
    public static final byte ASCII_CODE_ENTER        = 0x0D;
    public static final int  POLLING_RATE            = 10000;   // 10 seconds polling rate

    /** localized output messages */
    private ResourceBundle messages;

    private SimpleReadOnlyDeviceAttribute<Float> temperature;

    private Logger logger;
    private String hardwareId;
    private UART rs232;

    /**
     * @param logger logger associated with this endpoint
     */
    @Inject
    public Rs232ThermometerEndpoint(Logger logger) {
        this.logger = logger;
        messages = ResourceBundle.getBundle("Messages");
        logger.log(Level.INFO, "Created new RS-232 thermometer endpoint.");
    }

    /**
     * Initialize this endpoint with a serial port input stream
     *
     * @param hardwareId a globally unique hardware ID.
     * @param uart       the rs232 stream to use for communication with the device
     */
    void init(String hardwareId, UART uart) {
        this.hardwareId = hardwareId;
        rs232 = uart;

        /* Set the polling rate to POLLING_RATE. The poll method will be called to read the data from the buffer */
        setPollingRate(Duration.ofMillis(POLLING_RATE));

        temperature = new SimpleReadOnlyDeviceAttribute<>(
                this, "temperature", getEndpointContext().getEventService());
        logger.info("Starting RS-232 thermometer listener...");
    }

    /**
     * Processes a reading from the RS-232 thermometer and sends the measurement message
     * to the server.
     *
     * @param t the temperature data read from the RS-232 thermometer
     */
    private void processTemperatureReading(Float t) {
        try {
            System.out.printf(messages.getString("temperature"), getEndpointContext().getEndpointId(),t);
            
            getEndpointContext().getMessagingService().submit(new ThermometerEvent.Builder(this)
                    .temperature(t)
                    .build().toDataMessage());
            temperature.notifyValueUpdated(t);
        } catch(Exception e) {
            logger.log(Level.FINEST, "Invalid temperature value: {0}", t);
        }
    }

    /**
     * Method that is invoked to do the actual polling.
     *
     */
    @Override
    protected void poll() {
        logger.log(Level.INFO, "=======Papouch TM digital temperature sensor RS-232========");
        double temp = TemperatureDeviceRTU.readUARTRS232(this.rs232, logger);
        processTemperatureReading((float)temp);
    }

    /**
     * Device framework resource access method.
     *
     * @return the temperature property for this endpoint
     */
    @IoTResource
    public ReadOnlyDeviceAttribute<Float> temperatureProperty() {
        return temperature;
    }

    @Override
    public Float getTemperature() {
        return temperature.getValue();
    }

    private static class TemperatureDeviceRTU
    {
        /**
         * Parse the incoming response from Papouch TM digital temperature sensor.
         * The data are transmitted in a simple ASCII protocol within which the temperature values are given
         * directly in degrees Celsius. The sensor is connected via a standard RS232 serial port.
         * When DTR signal remains active, the TM sensor measures and sends the temperature value every 10 seconds.
         * Output: <sign><3 characters-integers degress Celsius><decimal point><1 character-tenths of degree Celsius><C><Enter>
         * E.g.: +025.3C
         *
         * For the sake of simplicity this sample code implement minimal error handling. It does not handle the case
         * when temperature sensor provides data faster than polling rate(this means accumulation of temperature data
         * in UART buffer) or slower(this means temperature error value 0xFFFF reported). It expects sensor to send temperature
         * value precisely every 10 seconds (DTR signal remains active). A better alternative to this approach would be
         * to activate DTR signal each time before temperature reading.
         */
        private static double readUARTRS232(UART rs232, Logger logger) {
            double temperatureVal = 0xFFFF;
            double tmp;	//temporary variable for temperature value
            byte   cnt;
            boolean tempSensErr;
            int    bytesRead;

            logger.log(Level.INFO, "Reading from RS-232");
            try {
		        /*
		        Read output data from temperature sensor. It should contain 8 characters with following format:
		        <sign><3 characters-integers degress Celsius><decimal point><1 character-tenths of degree Celsius><C><Enter>
		        */
                ByteBuffer sensorData = ByteBuffer.allocateDirect(SENSOR_OUTPUT_BYTES_NUMBER);
                bytesRead = rs232.read(sensorData);
                tempSensErr = true;

                // Parse temperature sensor output data. If it does not fit into expected format then return 0xFFFF
		        if (SENSOR_OUTPUT_BYTES_NUMBER == bytesRead) {
                    if (ASCII_CODE_SIGN == sensorData.get(SENSOR_OUTPUT_SIGN)) {
                        tmp = 0;
                        for(cnt=SENSOR_OUTPUT_TEMP_VALUE; cnt<SENSOR_OUTPUT_DECIMAL_POINT; cnt++) {
                            tmp = tmp*10 + sensorData.get(cnt) - ASCII_CODE_ZERO;
                        }

                        if (ASCII_CODE_DECIMAL_POINT == sensorData.get(SENSOR_OUTPUT_DECIMAL_POINT)) {
                            tmp = tmp*10 + sensorData.get(SENSOR_OUTPUT_TEENTH_TEMP_VALUE) - ASCII_CODE_ZERO;
                            tmp = tmp/10;

                            if ( (ASCII_CODE_CELSIUS == sensorData.get(SENSOR_OUTPUT_CELSIUS)) &&
                                 (ASCII_CODE_ENTER   == sensorData.get(SENSOR_OUTPUT_ENTER)) ) {
                                // Output data provided by temperature sensor fits into expected format
                                temperatureVal = tmp;
                                tempSensErr = false;
                            }
                        }
		            }
                }

                // Sensor output data might not fit into expected format due to various reasons (lost bytes, etc).
                // Implement error handling: flush(if not empty)/poll(if empty) UART buffer until "Enter" ASCII code
                // is found. "Enter" ASCII character is marking end of temperature output data.
                // For erroneous UART readings, the returned temperature value is 0xFFFF.
                if (tempSensErr) {
                    ByteBuffer dummy = ByteBuffer.allocateDirect(1);
                    do {
                        rs232.read((ByteBuffer)dummy.clear());
                    }
                    while (ASCII_CODE_ENTER != dummy.get(0));

                    logger.log(Level.INFO, "Read temperature sensor output data error!");
                }

            } catch (IOException e) {
                logger.log(Level.WARNING, "An IOException was caught :"+e.getMessage());
            }

            logger.log(Level.INFO, "Measured temperature: " + Double.toString(temperatureVal));
            return temperatureVal;
        } //end read
    }
}
