/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.daf.type.ibeacon;

import oracle.iot.endpoint.DataEvent;
import oracle.iot.event.EventType;
import oracle.iot.message.DataMessage;
import oracle.iot.util.ToStringBuilder;

import java.time.Instant;

public class IBeaconEvent<T extends IBeaconEndpoint> extends DataEvent.Default<T> {
    private static final String FORMAT_URN = "urn:com:oracle:iot:device:location:ibeacon:attributes";

    private Float ora_rssi = null;
    private Integer ora_txPower = null;

    
    public final Float getOra_rssi() {
        return ora_rssi;
    }

    public final Integer getOra_txPower() {
        return ora_txPower;
    }

    public static final EventType<IBeaconEvent<? extends IBeaconEndpoint>> REPORT =
            new EventType<>("IBEACON", DataEvent.ANY);

    protected IBeaconEvent(final T source,
                                   final EventType<? extends IBeaconEvent> type,
                                   final Instant timestamp)
    {
        super(source, type, FORMAT_URN, timestamp);
    }

    @Override
    protected DataMessage.Builder constructDataMessage(final DataMessage.Builder builder) {
        DataMessage.Builder out = builder;
        
        if (ora_rssi != null) builder.dataItem("ora_rssi", ora_rssi);
        if (ora_txPower != null) builder.dataItem("ora_txPower", ora_txPower);
        return out;
    }

    @Override
    protected ToStringBuilder constructToString(ToStringBuilder builder) {
        ToStringBuilder out = super.constructToString(builder);

        if (ora_rssi != null) out.append("ora_rssi", ora_rssi);
        if (ora_txPower != null) out.append("ora_txPower", ora_txPower);
        return out;
    }

    @Override
    public IBeaconEvent<T> copy(final T newSource) {
        IBeaconEvent event = new IBeaconEvent(newSource, IBeaconEvent.REPORT, getTimestamp());
        event.ora_rssi = ora_rssi;
        event.ora_txPower = ora_txPower;
        return event;
    }


    @Override
    public String toString() {
        return constructToString(new ToStringBuilder("IBeaconEvent")).toString();
    }

    public static class Builder<T extends IBeaconEndpoint> {
        T source;
        Instant timestamp;
        private Float ora_rssi = null;
        private Integer ora_txPower = null;

        public Builder(final T source) { this.source = source; this.timestamp = Instant.now(); }
        public Builder(final T source, final Instant timestamp) { this.source = source; this.timestamp = timestamp;
    }

        public Builder ora_rssi(Float ora_rssi) { this.ora_rssi = ora_rssi; return this; }
        public Builder ora_txPower(Integer ora_txPower) { this.ora_txPower = ora_txPower; return this; }

        public IBeaconEvent build() {
            IBeaconEvent event = new IBeaconEvent(source, IBeaconEvent.REPORT, timestamp);
            event.ora_rssi = ora_rssi;
            event.ora_txPower = ora_txPower;
            return event;
        }
    }
}
