/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.bluetoothle;

import java.lang.InterruptedException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.Map;

import com.oracle.iot.sample.adapter.bluetoothle.BluetoothLEEndpoint;
import jdk.bluetooth.BluetoothIOException;
import jdk.bluetooth.RemoteDevice;
import oracle.iot.beans.property.IntegerProperty;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.Metadata;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import com.oracle.iot.sample.daf.type.heartratemonitor.HeartRateMonitorEndpoint;
import com.oracle.iot.sample.daf.type.heartratemonitor.HeartRateMonitorEvent;
import oracle.iot.event.EventService;

import com.oracle.bluetooth.le.HeartRateCollector;


import javax.inject.Inject;
import java.util.logging.Logger;

@IoTDeviceEndpoint
public class HeartRateEndpoint extends BluetoothLEEndpoint implements HeartRateMonitorEndpoint {
    HeartRateCollector collector;
    private IntegerProperty heartRateProp;
    private final SimpleReadOnlyDeviceAttribute<Integer> heartRate;
    protected Logger logger;
    protected Metadata metadata;
    protected String address;
    protected String hardwareId;
    protected EventService eventService;
    private Future<Void> connectFuture;

    @Inject
    public HeartRateEndpoint(EventService es) {
        super();
        eventService = es;
        heartRate = new SimpleReadOnlyDeviceAttribute<>(this, "heartRate", es);
    }

    @Override
    public ReadOnlyDeviceAttribute<Integer> heartRateProperty() {
        return heartRate;
    };

    @Override
    public Integer getHeartRate() {
        return heartRate.getValue();
    };

    /**
     * Initiate colleciton of heart rate data from RemoteDevice
     *
     * @param device the remote heart rate profile device
     */
    public void connect(RemoteDevice device) {
        try {
            collector = HeartRateCollector.getInstance();

            logger.info("Connecting to device: " + device.getAddress());
            Consumer<Map<String, Object>> dataConsumer =
                    (Map<String, Object> measurements) -> dataReceived(measurements);
            connectFuture = collector.connect(device, dataConsumer);

            try {
                connectFuture.get();
            } catch (ExecutionException | InterruptedException e2) {
                e2.printStackTrace();
            }
        } catch (BluetoothIOException e) {
            e.printStackTrace();
        }
    }

    public void disconnect() {
        // Heart rate profile does not currently support disconnect
    }

    /**
     * Initializes  metadata associated with this endpoint.
     *
     * @param hardwareId a globally unique ID for the associated connected device.  For example, the Bluetooth address.
     * @param metadata   the IoT framework metadata associated with this device
     * @param address    the bluetooth address for the associated connected device
     * @param logger the logger to use for this device
     */
    public void init(String hardwareId, Metadata metadata, String address, Logger logger)
    {
        this.metadata = metadata;
        this.address = address;
        this.hardwareId = hardwareId;
        this.logger = logger;
    }

    public void dataReceived(Map<String, Object> measurements) {
        Integer location;
        int heartRateVal = (Integer)measurements.get(HeartRateCollector.KEY_VALUE);

        logger.info("received heart rate measurement <" + heartRateVal +"> from " + address);
        this.heartRate.notifyValueUpdated(heartRateVal);
        
        eventService.fire(new HeartRateMonitorEvent.Builder(this, this.heartRate.getLastUpdateTime())
            .heartRate(heartRateVal)
            .build());

    }

}
