/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.bluetoothle;


import jdk.bluetooth.BluetoothException;
import jdk.bluetooth.LocalDevice;
import jdk.bluetooth.RemoteDevice;
import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;
import oracle.iot.device.Metadata;

import javax.inject.Inject;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Device adapter for Bluetooth LE capable devices. This is an experimental implementation which reflects the current
 * state of the BlueZ and Bluetooth stack implementations. Ultimately, this may be replaced with the a more generic
 * GATT based implementation when support is available.
 */
@IoTDeviceAdapter
public class BluetoothLEAdapter extends AbstractDeviceAdapter {

    protected Logger logger;

    private HashMap<String, Class> deviceClasses = new HashMap<>();

    /**
     * Creates a new instance of the {@code BluetoothLEAdapter}.
     *
     * @param logger Logger associated with this endpoint.
     */
    @Inject
    public BluetoothLEAdapter(Logger logger) {
        this.logger = logger;
    }


    private void initDeviceClasses() {
        deviceClasses.put("MIO GLOBAL", HeartRateEndpoint.class);
        deviceClasses.put("HTC Fetch", ProximityEndpoint.class);

        for (HashMap.Entry<String, Class> entry : deviceClasses.entrySet()) {
            logger.log(Level.FINE, " Bluetooth LE Device added to search: " + entry.getKey() + " : " + entry.getValue());
        }
    }

    @Override
    protected void start() throws Exception {
        logger.log(Level.FINE, "Bluetooth LE adapter created.");
        initDeviceClasses();
        new Thread(() -> {
            RemoteDevice remoteDevice = null;
            do {
                synchronized (this) {
                    try {
                        this.wait(5000);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                for (HashMap.Entry<String, Class> entry : deviceClasses.entrySet()) {
                    logger.log(Level.FINE, "Searching for bluetooth LE device: " + entry.getKey());
                    remoteDevice = findDevice(entry.getKey());

                    if (remoteDevice != null) {
                        connectEndpoint(remoteDevice, entry.getValue());
                        logger.log(Level.FINE, "Bluetooth LE device found " + entry.getKey());
                        break;
                    }

                }
            } while (remoteDevice == null);

        }
        ).start();

    }

    protected void connectEndpoint(RemoteDevice device, Class cls) {
        // gather metadata
        String name = device.getName();
        String addr = device.getAddress();
        String manufacturer = Integer.toString(device.getManufacturerID());

        if(manufacturer.equals("-1")) {
            manufacturer = "65535"; //This value is reserved as the default vendor ID when no Device ID service record is present in a remote device.
        }


        logger.log(Level.FINE, "registering BluetoothLEEndpoint: name[" + name + "] addr[" + addr + "] "
                + "manufacturer[" + manufacturer + "] productId:[" + device.getProductID() + "] vendorId:["
                + device.getVendorSourceID() + "]");

        // instantiate metadata class
        Metadata metadata = Metadata.builder().
                protocol("bluetooth").
                protocolDeviceClass("le").
                protocolDeviceId(addr).
                manufacturer(manufacturer).
                deviceClass(cls.getSimpleName()).
                serialNumber(addr).build();


        create(device, metadata, addr, cls);
    }

    /**
     * Registers a connected device with the device framework and, if successful, initiates remote connection
     *
     * @param device   the remote bluetooth LE device representing the remote device
     * @param metadata device framework metadata to associated with connected device
     * @param addr     bluetooth address for the connected device
     */
    public void create(RemoteDevice device, Metadata metadata, String addr, Class cls) {
        logger.log(Level.INFO, "Creating Bluetooth LE Endpoint framework device");
        // hardwareId is the Bluetooth address, which is globally unique and will always be the same value.
        String hardwareId = addr;
        ObservableFuture future = registerDevice(hardwareId, metadata, cls, null);

        future.addListener((observable) -> {
            try {
                BluetoothLEEndpoint endpoint = (BluetoothLEEndpoint) future.get();
                endpoint.init(hardwareId, metadata, addr, logger);
                logger.log(Level.INFO, "Bluetooth LE Endpoint registered: " + endpoint.getId());
                endpoint.connect(device);
            } catch (final Exception e) {
                e.printStackTrace();
            }
        });
    }

    private RemoteDevice findDevice(String name) {
        Collection<RemoteDevice> devices;

	// Bluetooth device names sometimes contain trialing white spaces - remove when searching
	name = name.trim();

        try {
            try (LocalDevice localDevice = LocalDevice.getDefault()) {
                devices = localDevice.getPairedDevices();
                if (devices.isEmpty()) {
                    return null;
                }
                for (RemoteDevice device : devices) {
                    logger.log(Level.FINE, "Device found: " + device.getName().trim());
                    if (device.getName().trim().equals(name)) {
                        return device;
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        } catch (BluetoothException e) {
            e.printStackTrace();
        }
        return null;
    }
}

