/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.health;

import com.oracle.iot.sample.daf.adapter.health.PersonalHealthEndpoint;
import jdk.phd.Phd;
import jdk.phd.report.ConfigReport;
import jdk.phd.report.ScanReport;
import jdk.phd.report.spec.PulseOximeter;
import oracle.iot.beans.property.DoubleProperty;
import oracle.iot.beans.property.IntegerProperty;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;

import com.oracle.iot.sample.daf.type.pulseoximeter.PulseOximeterEvent;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.time.Instant;
import java.util.logging.Level;

@IoTDeviceEndpoint
public class PulseOximeterEndpoint extends PersonalHealthEndpoint implements com.oracle.iot.sample.daf.type.pulseoximeter.PulseOximeterEndpoint {

    private final SimpleReadOnlyDeviceAttribute<Float> oxygenSaturation;
    private final SimpleReadOnlyDeviceAttribute<Integer> heartRate;
    private DoubleProperty oxygenSaturationProp;
    private IntegerProperty heartRateProp;

    @Inject
    public PulseOximeterEndpoint(EventService es)
    {
        super();
        oxygenSaturation = new SimpleReadOnlyDeviceAttribute<>(this, "Pulse oximeter oxygen saturation", es);
        heartRate = new SimpleReadOnlyDeviceAttribute<>(this, "Pulse oximeter heart rate",es);
    }

    public ReadOnlyDeviceAttribute<Float> oxygenSaturationProperty() {
        return oxygenSaturation;
    };

    public Float getOxygenSaturation() {
        return oxygenSaturation.getValue();
    };

    public ReadOnlyDeviceAttribute<Integer> heartRateProperty() {
        return heartRate;
    };

    public Integer getHeartRate() {
        return heartRate.getValue();
    };

    @Override
    public void scanReported(Phd phd, ScanReport scanReport) {
        logger.log(Level.FINE, "PhdHandler.scanReported");

        PulseOximeter poScanReport = (PulseOximeter)scanReport;

        if(poScanReport.getPulse() == null || poScanReport.getPulse().size() == 0)
            return;

        for(int index = 0; index < poScanReport.getPulse().size(); index++)
        {
            dispatch(poScanReport.getPulse().get(index).getTimestamp(), (int) poScanReport.getPulse().get(index).getValue(), (float)poScanReport.getSpo2().get(index).getValue());
        }
    }

    @Override
    public boolean configReported(Phd phd, ConfigReport configReport) {
        logger.log(Level.FINE, "PhdHandler.configReported");
        if(configReport.getConfig().getConfigId() == 0x191) {
            logger.log(Level.INFO, "accepted streaming configuration from pulse oximeter device: " + address);
            return true;
        }
        logger.log(Level.FINE, "PulseOximeterEndpoint Config " + configReport.getConfig().getConfigId() + " reported - rejecting.");
        return false;
    }

    void dispatch(long time, int heartRate, float oxygenSaturation) {
        logger.log(Level.FINE, "generating event: (" + heartRate + ":" + oxygenSaturation +")  from device: " + address);

        this.oxygenSaturation.notifyValueUpdated(oxygenSaturation);
        this.heartRate.notifyValueUpdated(heartRate);

        final EventService es = getEndpointContext().getEventService();
        es.fire(new PulseOximeterEvent.Builder(this)
                .heartRate(heartRate)
                .oxygenSaturation(oxygenSaturation)
                .build());
    }
}

