/*
 * Copyright (c) 2014, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
package com.oracle.iot.sample.daf.adapter.bluetooth;


import com.oracle.iot.sample.daf.adapter.health.PersonalHealthEndpoint;

import java.util.Collection;
import java.util.HashMap;

/**
 * Base class for the Bluetooth health device database.
 * The registry contains all known connected Personal Health Devices.
 * For future implementations, these may be registered with the
 * local gateway, queried from the server, or collected at run time as health devices are connected.
 */
public class EndpointRegistry {
    HashMap<String, PersonalHealthEndpoint> endpoints = new HashMap<>();
    private static EndpointRegistry registry = null;

    /**
     * Public factory method. Returns the default implementation of the endpoint registry.
     *
     * @return the default endpoint registry implementation, never be {@code null}
     */
    public static synchronized EndpointRegistry getInstance()
    {
        if(registry == null) registry = new EndpointRegistry();
        return registry;
    }

    /**
     * Returns all known bluetooth health endpoints.
     *
     * @return a list of all known bluetooth health endpoints.
     */
    public Collection<PersonalHealthEndpoint> getEndpoints() {
        return endpoints.values();
    }

    /**
     * Returns a bluetooth health endpoint by bluetooth address, or null if the address is not associated
     * with a known bluetooth health endpoint.
     *
     * @param address the bluetooth address of the device
     *
     * @return the bluetooth health endpoint associated with the address, or {@code null} if endpoints contains no mapping for address
     */
    public PersonalHealthEndpoint getEndpoint(String address) {
        return endpoints.get(address);
    }

    /**
     * Associates a bluetooth health endpoint with bluetooth address
     *
     * @param address the bluetooth address of the device
     * @param endpoint the bluetooth health endpoint to associate with the address.
     */
    public void addEndpoint(String address, PersonalHealthEndpoint endpoint) {
        endpoints.put(address, endpoint);
    }
}
