/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import com.oracle.iot.sample.daf.type.ibeacon.IBeaconEndpoint;
import com.oracle.iot.sample.daf.type.ibeacon.IBeaconEvent;
import oracle.iot.device.AbstractDeviceEndpoint;
import oracle.iot.device.IoTDeviceEndpoint;
import oracle.iot.device.attribute.ReadOnlyDeviceAttribute;
import oracle.iot.device.attribute.SimpleReadOnlyDeviceAttribute;
import oracle.iot.event.EventService;

import javax.inject.Inject;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;


/**
 * Simulates a beacon which is being moved away from a Bluetooth device which is detecting the beacon, then moving
 * closer, then repeating.
 */
@IoTDeviceEndpoint
public class IBeaconEndpointImpl extends AbstractDeviceEndpoint implements IBeaconEndpoint {
    private boolean isStarted = false;
    private final int MAX_DISTANCE = 50;
    private static final int NUM_STEPS = 5;
    private static final AtomicBoolean done = new AtomicBoolean(false);
    private  boolean isInitialized = false;

    private boolean isOn;
    private int currentStep = 0;
    private int stepDirection = 1;
    private int stepDistance = 0;

    private Float ora_rssiValue;
    private Integer ora_txPowerValue = -74;

    private SimpleReadOnlyDeviceAttribute<Float> ora_rssi;
    private SimpleReadOnlyDeviceAttribute<Integer> ora_txPower;
    private Thread advertisementThread = null;


    @Inject
    public IBeaconEndpointImpl(Logger logger) {
        super();
    }

    @Override public ReadOnlyDeviceAttribute<Float> ora_rssiProperty() { return ora_rssi; }
    @Override public ReadOnlyDeviceAttribute<Integer> ora_txPowerProperty() { return ora_txPower; }

    private void calculateNewStepDistance() {
        Random random  = new Random();
        int num = (random.nextInt(MAX_DISTANCE * 2) - MAX_DISTANCE) % NUM_STEPS;
        // Step should never be negative or 0.
        stepDistance = (num < 0) ? -num : (num == 0) ? 5 : num;
    }

    private float calculateNextValue() {
        float nextValue = BeaconDeviceSampleAdapter.HOME_VALUE - (currentStep * stepDistance);
        currentStep = currentStep + stepDirection;
        return nextValue;
    }

    /**
     * Initializes the device endpoint.
     */
    void init() {
        EventService eventService = getEndpointContext().getEventService();
        ora_rssi = new SimpleReadOnlyDeviceAttribute<>(this, "ora_rssi", eventService, () -> ora_rssiValue);
        ora_txPower = new SimpleReadOnlyDeviceAttribute<>(this, "ora_txPower", eventService, () -> ora_txPowerValue);
    }

    @Override
    protected void start() throws Exception {
        if (isStarted) { return; }
        isStarted = true;
        done.set(false);

        advertisementThread = new Thread(() -> {
            isOn = true;

            while (!done.get()) {
                synchronized (this) {
                    try {
                        this.wait(2000);
                    } catch (Exception e) {
                    }

                    if (isOn) {
                        if (currentStep == 0) {
                            stepDirection = 1;
                            calculateNewStepDistance();
                        } else if (currentStep == NUM_STEPS) {
                            stepDirection = -1;
                        }

                        ora_rssiValue = calculateNextValue();
                        IBeaconEvent.Builder event = new IBeaconEvent.Builder(this);
                        event.ora_rssi(ora_rssiValue);
                        if (!isInitialized) {
                            event.ora_txPower(ora_txPowerValue);
                            isInitialized = true;
                        }
                        getEndpointContext().getEventService().fire(event.build());
                    }
                }
            }
        });

        advertisementThread.start();
    }

    @Override
    protected void stop() throws Exception {
        done.set(true);
        isOn = false;

        if (advertisementThread != null) {
            advertisementThread.notify();

            try {
                advertisementThread.join();
            } catch (InterruptedException e) {

            }
        }

        super.stop();
    }
}
