/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;


import oracle.iot.concurrent.ObservableFuture;
import oracle.iot.device.AbstractDeviceAdapter;
import oracle.iot.device.IoTDeviceAdapter;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.inject.Inject;


@IoTDeviceAdapter
public class BeaconDeviceSampleAdapter extends AbstractDeviceAdapter {
    static final int HOME_VALUE = -58;

    private Integer major = 10863;
    private Integer minor = 61653;
    protected final Logger logger;
    private String  iBeaconHardwareId = null;
    private String  eddystoneHardwareId = null;
    private String  iBeaconUuid = null;
    private String  eddystoneUuid = null;

    @Inject
    public BeaconDeviceSampleAdapter(Logger logger) {
        this.logger = logger;
        logger.log(Level.INFO, "Created new Beacon sample device adapter.");
    }

    @Override
    protected void start() throws Exception {
        // Get this device adapter's endpoint ID from the system.
        String endpointId = getEndpointContext().getEndpointId();
        this.iBeaconUuid = endpointId + "_Sample_iBeacon";
        this.eddystoneUuid = endpointId + "_Sample_Eddystone";
        this.iBeaconHardwareId = iBeaconUuid + ":" + major + ":" + minor;
        this.eddystoneHardwareId = eddystoneUuid + ":" + major + ":" + minor;
        boolean enableIBeacon;
        boolean enableEddystone;

        String enableIBeaconProp =  System.getProperty("com.oracle.iot.sample.enableIBeacon", "true");

        if ("true".equals(enableIBeaconProp)) {
            enableIBeacon = true;
        } else {
            enableIBeacon = false;
        }

        System.out.println("enableIBeaconProp=" + enableIBeaconProp);

        String enableEddystoneProp =  System.getProperty("com.oracle.iot.sample.enableEddystone", "false");

        if ("true".equals(enableEddystoneProp)) {
            enableEddystone = true;
        } else {
            enableEddystone = false;
        }

        System.out.println("enableEddystoneProp=" + enableEddystoneProp);

        if (enableIBeacon || enableEddystone) {
            new Thread(() -> {
                // Give time to initialize.
                synchronized (this) {
                    try {
                        this.wait(5000);
                    } catch (Exception e) {
                    }
                    ;
                }

                if (enableIBeacon) {
                    final String address = "iBeacon:" + endpointId;
                    System.out.println("address:" + address);

                    IBeaconMetadata iBeaconMetadata = IBeaconMetadata.builder()
                        .deviceClass("LE")
                        .major("Major: " + major)
                        .manufacturer("Estimote")
                        .minor("Major: " + minor)
                        .protocol("Bluetooth-LE")
                        .protocolDeviceId(address)
                        .protocolDeviceClass("iBeacon")
                        .serialNumber("Serial: " + address)
                        .uuid("UUID: " + iBeaconUuid)
                        .build();

                    ObservableFuture<IBeaconEndpointImpl> iBeaconFuture = registerDevice(
                        iBeaconHardwareId,
                        iBeaconMetadata, IBeaconEndpointImpl.class,
                        (iBeaconEndpoint) -> iBeaconEndpoint.init());

                    // Listen for the new device representation being ready
                    iBeaconFuture.addListener((observable) -> {
                        try {
                            IBeaconEndpointImpl iBeaconEndpoint = iBeaconFuture.get();
                            logger.log(Level.FINE, "iBeacon registered: " + address);
                        } catch (final Exception e) {
                            e.printStackTrace();
                        }
                    });
                } else {
                    System.out.println("The IBeacon beacon is disabled.");
                }

                if (enableEddystone) {
                    final String eddystoneAddress = "Eddystone Beacon:" + endpointId;
                    EddystoneBeaconMetadata eddystoneBeaconMetadata = EddystoneBeaconMetadata.builder()
                        .deviceClass("LE")
                        // TODO: Get manufacturer name from device.
                        .manufacturer("Estimote")
                        .protocol("Bluetooth-LE")
                        .protocolDeviceId(eddystoneAddress)
                        .protocolDeviceClass("eddystone")
                        .serialNumber("Serial:" + eddystoneAddress)
                        .uuid("UUID: " + eddystoneUuid)
                        .build();

                    ObservableFuture<EddystoneBeaconEndpointImpl> eddystoneFuture = registerDevice(
                        eddystoneHardwareId,
                        eddystoneBeaconMetadata,
                        EddystoneBeaconEndpointImpl.class,
                        (eddystoneBeaconEndpoint) -> eddystoneBeaconEndpoint.init());

                    // Listen for the new device representation being ready.
                    eddystoneFuture.addListener((observable) -> {
                        try {
                            EddystoneBeaconEndpointImpl eddystoneBeaconEndpoint = eddystoneFuture.get();
                            logger.log(Level.FINE, "Eddystone Beacon registered: " + eddystoneBeaconEndpoint.getId());
                        } catch (final Exception e) {
                            e.printStackTrace();
                        }
                    });
                } else {
                    System.out.println("The Eddystone beacon is disabled.");
                }
            }).start();
        } else {
            System.out.println("The IBeacon and Eddystone beacons are disabled.");
        }
    }
}
