/*
 * Copyright (c) 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import java.util.Arrays;


/* This is a helper class that implements the smoothing/averaging the RSSI values
   recevied from a Beacon device
 */
public class RssiSmoother {
    /**
     * Number of data points to average over for RSSI smoothing
     */
    private static final int WINDOW_SIZE = 20;

    /**
     * Interval to wait between messages sent to the server, measured in number of data points
     * received from the beacon
     */
    private static final int SEND_INTERVAL = 3;

    int[] rssis;
    int rssiCount;
    int rssi;
    boolean readyToSend = false;
    int rssiRaw;
    float ora_txPowerValue;

    RssiSmoother() {
        this.rssis = new int[WINDOW_SIZE];
        this.rssiCount = 0;
        this.ora_txPowerValue = 0;
    }

    void setTxPower(float txPower) {
        this.ora_txPowerValue = txPower;
    }


    private int smoothData(int count, int[] data) {
        int average;
        int len = (count < data.length) ? count: data.length;
        if (len < 1)
            return 0;

        //copy the data to include in the calculation
        int[] tempData = Arrays.copyOfRange(data, 0, len);
        Arrays.sort(tempData);

        //calculate the mean of the middle 80% of data
        int sum = 0;
        int discard = (int) Math.round(0.1*len);
        for (int i=discard; i<len-discard; i++) {
            sum += tempData[i];
        }
        average = (int)(sum*1.0/(len-2*discard));
        return average;
    }

    public void setRssi(int rssi) {
        this.rssi = rssi;
        //Put the new rssi value at the next index in the array
        this.rssis[rssiCount++ % WINDOW_SIZE] = rssi;
        if (rssiCount == Integer.MAX_VALUE) {
            rssiCount = Integer.MAX_VALUE % WINDOW_SIZE+WINDOW_SIZE;
        }
        //Determine if it is time to send out the average rssi value yet
        if (rssiCount%SEND_INTERVAL == SEND_INTERVAL-1) {
            readyToSend = true;
        }
        this.rssiRaw = rssi;
    }

    public int getRssi() {
        return smoothData(rssiCount, rssis);
    }

    public double getDistance() {
        return calculateDistance(getRssi());
    }

    private double calculateDistance(int rssi) {
        if (rssi == 0) {
            return -1.0; // cant determine
        }
        double ratio = rssi * 1.0 / ora_txPowerValue;
        if (ratio < 1.0) {
            return Math.pow(ratio, 10);
        } else {
            double accuracy = (0.89976) * Math.pow(ratio, 7.7095) + 0.111;
            // 0.89976, 7.7095 and 0.111 are the three constants calculated when solving for a best fit curve to our measured data points.
            return accuracy;
        }
    }

    public boolean isReadyToSend() {
        if (readyToSend){
            readyToSend = false;
            return true;
        }
        return false;
    }
}
