/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and the
 * Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

package com.oracle.iot.sample.adapter.beacon;

import oracle.iot.device.Metadata;
import oracle.iot.messaging.IoTResource;
import oracle.iot.module.IoTExport;

/**
 *  This class represents the Meta-data associated with the Eddystone beacon devices.
 */
@IoTExport
public interface EddystoneBeaconMetadata extends Metadata {

    /** This represents Namespace Identifier for the Eddystone beacon. */
    @IoTResource
    String getNamespaceId();

    /** This represents Instance Identifier for the Eddystone beacon. */
    @IoTResource
    String getInstanceId();

    /** This represents the UUID of the Eddystone beacon. */
    @IoTResource
    String getUuid();

    static Builder<?> builder() {
        return new Builder();
    }

    class Default extends Metadata.Default implements EddystoneBeaconMetadata {
        private static final String NAMESPACEID = "namespaceid";
        private static final String INSTANCEID = "instanceid";
        private static final String UUID  = "uuid";

        protected Default(EddystoneBeaconMetadata.Builder<?> builder) {
            super(builder);
        }

        @Override
        public String getNamespaceId() {
            return (String) getAttribute(NAMESPACEID);
        }

        @Override
        public String getInstanceId() {
            return (String) getAttribute(INSTANCEID);
        }

        @Override
        public String getUuid() {
            return (String) getAttribute(UUID);
        }
    }

    class Builder<T extends Builder<T>> extends Metadata.Builder<T> {
        protected Builder() {
        }

        @Override
        public EddystoneBeaconMetadata build() {
            return new Default(this);
        }

        @SuppressWarnings("unchecked")
        public final T major(String namespaceId) {
            attributes.put(Default.NAMESPACEID, namespaceId);
            return (T) this;
        }

        @SuppressWarnings("unchecked")
        public final T minor(String instanceId) {
            attributes.put(Default.INSTANCEID, instanceId);
            return (T) this;
        }
        @SuppressWarnings("unchecked")
        public final T uuid(String uuid) {
            attributes.put(Default.UUID, uuid);
            return (T) this;
        }
    }
}
