/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include "mbed.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "SDFileSystem.h"
#include "bootstrapper.h"

#ifndef IOTCS_MBED_TS_PASSWORD
#define IOTCS_MBED_TS_PASSWORD changeit\0
#endif // #ifndef IOTCS_MBED_TS_PASSWORD

#ifndef IOTCS_MBED_TS_PATH
#define IOTCS_MBED_TS_PATH /sd/trusted_assets_store\0
#endif // #ifndef IOTCS_MBED_TS_PATH

#define __STRINGIFY(x) #x
#define STRINGIFY(x) __STRINGIFY(x)

#define CMD_SET_TIME   (0x73)  // 's'

Serial pc(USBTX, USBRX);
EthernetInterface eth;
NTPClient ntp;

DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);
DigitalOut led_blue(LED_BLUE);

const char *mbed_ts_path = STRINGIFY(IOTCS_MBED_TS_PATH);
const char *mbed_ts_password = STRINGIFY(IOTCS_MBED_TS_PASSWORD);

#ifndef IOTCS_SDCARD_MOUNT_NAME
#define IOTCS_SDCARD_MOUNT_NAME sd\0
#endif // #ifndef IOTCS_SDCARD_MOUNT_NAME
// mount sd card with IOTCS_SDCARD_MOUNT_NAME name
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, STRINGIFY(IOTCS_SDCARD_MOUNT_NAME)); // MOSI, MISO, SCK, CS

/**
 * Type of trusted assets store
 * 0 - old
 * 1 - unified
 */
static int tas_type = OLD_TRUSTED_ASSETS_STORE_FORMAT;
static trusted_assets_store store;
static char buffer[REQUEST_RECEIVE_BUF_SIZE];

static char property_format[PROP_FORMAT_SIZE];
static char line_buffer[PROP_FORMAT_SIZE];

static bs_result read_property(FILE* file, const char* prop_name, char* prop_value) {
    GOTO_ERR(NULL == fgets(line_buffer, TAM_MAX_PROP_SIZE, file));
    GOTO_ERR(0 >= snprintf(property_format, PROP_FORMAT_SIZE, "%s=%%%us", prop_name, TAM_MAX_PROP_SIZE - 1));
    GOTO_ERR(0 == sscanf(line_buffer, property_format, prop_value));
    pc.printf("    %s=%s\n", prop_name, prop_value);
    return BS_RESULT_OK;
error:
    pc.printf("Failed to read %s from trusted assets store\n", prop_name);
    return BS_RESULT_FAIL;
}

static bs_result read_trusted_assets_store(trusted_assets_store* store) {
    FILE* store_file;
    int version;
    pc.printf("Reading trusted asset file at %s\n", store->ta_store_name);

    store_file = fopen(store->ta_store_name, "r");
    if (store_file == NULL) {
        pc.printf("No trusted assets file found\n");
        return BS_RESULT_FAIL;
    }

    GOTO_ERR(EOF == (version = fgetc(store_file)));

    if (version != DEFAULT_FORMAT_VERSION) {
        tas_type = OLD_TRUSTED_ASSETS_STORE_FORMAT;
        rewind(store_file); /* put back first character */
        GOTO_ERR(read_property(store_file, "client.id", store->client_id));
        GOTO_ERR(read_property(store_file, "client.secret", store->shared_secret));
        GOTO_ERR(read_property(store_file, "trustAnchors", store->trust_store_type));

        if (0 == strcmp(store->trust_store_type, IN_PLACE_CA)) {
            GOTO_ERR(0 >= snprintf(property_format, PROP_FORMAT_SIZE, "%s=%%%us", "server.host", TAM_MAX_PROP_SIZE - 1));

            /* skip cert - everything between -----BEGIN CERTIFICATE----- and -----END CERTIFICATE----- */
            do {
                GOTO_ERR(NULL == fgets(line_buffer, PROP_FORMAT_SIZE, store_file));
            } while (0 != strncmp(line_buffer, "server.host", strlen("server.host")));
            GOTO_ERR(sscanf(line_buffer, property_format, store->server_host) != 1);
            pc.printf("    server.host=%s\n", store->server_host);
        } else {
            if (0 != strcmp(store->trust_store_type, SYSTEM_DEFAULT_CA)) {
                pc.printf("'trustAnchors' property has invalid value: expected " SYSTEM_DEFAULT_CA " or " IN_PLACE_CA "\n");
                goto error;
            }
            GOTO_ERR(read_property(store_file, "server.host", store->server_host));
        }

        GOTO_ERR(read_property(store_file, "server.port", store->server_port));
        GOTO_ERR(read_property(store_file, "server.scheme", store->server_scheme));
        GOTO_ERR(0 >= snprintf(property_format, PROP_FORMAT_SIZE, "%s=%%%us", "signature", TAM_MAX_PROP_SIZE - 1));

        /* skip indirect connected data*/
        do {
            GOTO_ERR(NULL == fgets(line_buffer, PROP_FORMAT_SIZE, store_file));
        } while (0 != strncmp(line_buffer, "signature", strlen("signature")));
        GOTO_ERR(sscanf(line_buffer, property_format, store->signature) != 1);
        pc.printf("    signature=%s\n", store->signature);
    } else {
        tas_type = UNIFIED_TRUSTED_ASSETS_STORE_FORMAT;
        pc.printf("Trusted assets store file has unified form. It means that it should be fully provisioned.\n");
    }

    fclose(store_file);
    return BS_RESULT_OK;

error:
    fclose(store_file);
    return BS_RESULT_FAIL;
}

static void handle_discover_request(UDPSocket *socket, Endpoint *client) {
    size_t offset = 0;
    buffer[offset++] = DISCOVER_RESPONSE;
    
    buffer[offset++] = 3;
    memcpy(buffer + offset, "MAC", 3);
    offset += 3;

    const char *mac = eth.getMACAddress();
    int mac_len = strlen(mac);
    
    buffer[offset++] = mac_len;
    memcpy(buffer + offset, mac, mac_len);
    offset += mac_len;

    if (socket->sendTo(*client, buffer, offset) < 0) {
        pc.printf("Failed to send discover response\n");
    }
}

static bs_result send_provision_response(UDPSocket *socket, Endpoint *client,
        char status) {
    char plain_data[] = {status};
    int plain_data_len = ARRAY_SIZE(plain_data);
    char data[ARRAY_SIZE(plain_data)];

    char buffer[ARRAY_SIZE(data) + 1];
    size_t buf_len = 0;

    buffer[0] = PROVISION_RESPONSE;
    memcpy(buffer + 1, plain_data, plain_data_len);
    buf_len = plain_data_len + 1;

    if (buf_len > 0 &&
            (socket->sendTo(*client, buffer, buf_len) < 0)) {
        perror("Failed to send provision response");
        return BS_RESULT_FAIL;
    }
    return BS_RESULT_OK;
}

static int bootstrap(void) {
    
    store.ta_store_name = mbed_ts_path;
    store.ta_store_pwd = mbed_ts_password;

    if (BS_RESULT_OK != read_trusted_assets_store(&store)) {
        Endpoint client;
        int buffer_len;
        UDPSocket socket;
        //socket.set_broadcasting(true);
        if (socket.bind(UDP_PORT)) {
            pc.printf("Failed to bind udp socket\n");
            return -1;
        }
        if (socket.join_multicast_group(MULTICAST_ADDRESS) != 0) {
            pc.printf("Error joining the multicast group\n");
        }

        pc.printf("Waiting for notification from provisioner\n");
        while (1) {
            // block waiting to receive a multicast packet
            if (0 > (buffer_len = socket.receiveFrom(client, buffer, sizeof (buffer)))) {
                pc.printf("recvfrom() failed\n");
                return -1;
            }
            if (buffer_len == 0)
                continue;
            // Discover or Provision?
            if (buffer[0] == DISCOVER_REQUEST) {
                handle_discover_request(&socket, &client);
                osDelay(2000);
            } else if (buffer[0] == PROVISION_REQUEST) {
                FILE* tas_file = fopen(store.ta_store_name, "w");

                if (!tas_file) {
                    pc.printf("Can't create %s file.\n", store.ta_store_name);
                    send_provision_response(&socket, &client,
                            STATUS_FAILURE);
                    return -1;
                }

                int value_length = (buffer[1] << 8) | buffer[2];

                if (value_length > buffer_len - 3) {
                    pc.printf("Not enough of memory. Expected: %d Read to buffer: %d. Please"
                            " increase REQUEST_RECEIVE_BUF_SIZE.\n", value_length, buffer_len - 3);
                    send_provision_response(&socket, &client,
                            STATUS_FAILURE);
                    fclose(tas_file);
                    return -1;
                }

                (0 < fwrite(&buffer[3], sizeof (char), buffer_len - 3, tas_file)) ?
                        send_provision_response(&socket, &client,
                        STATUS_SUCCESS) :
                        send_provision_response(&socket, &client,
                        STATUS_FAILURE);
                fclose(tas_file);
                break;
            } else {
                pc.printf("Unknown request received\n");
            }
        }
    }

    return 0;
}

extern "C" {
    extern int sample_main(const char* tam_path, const char* tam_password);
}

int main() {
    int status;
    // set serial baudrate    
    pc.baud(115200);
    // switch on blue led only
    led_red = 1;
    led_green = 1;
    led_blue = 0;

    // configure ethernet interface
    pc.printf("Init network...\n");
    if (eth.init() == 0 && eth.connect(20000) == 0) { //Use DHCP
        pc.printf("Network initialized\n");
        pc.printf("  IP Address:      %s\n", eth.getIPAddress());
        pc.printf("  Subnet mask:     %s\n", eth.getNetworkMask());
        pc.printf("  Gateway:         %s\n", eth.getGateway());
        pc.printf("  MAC-address:     %s\n", eth.getMACAddress());
        pc.printf("Trying to update time by NTP...\n");
        if (ntp.setTime(eth.getGateway()) == 0 || ntp.setTime("pool.ntp.org") == 0) {
            pc.printf("Set time successfully\n");
            time_t ctTime;
            ctTime = time(NULL);
            pc.printf("Time is set to (UTC): %s\n", ctime(&ctTime));
        } else {
            pc.printf("Can't get time from NTP\n");
        }
        pc.printf("Mount sd card\n");
        pc.printf("  Sd card name: %s\n\n", STRINGIFY(IOTCS_SDCARD_MOUNT_NAME));

        pc.printf("check if bootstrap is required...\n");
        if (bootstrap() != 0) {
            pc.printf("bootstrap failed\n");
            led_red = 0;
        } else {
            // Execute sample code
            status = sample_main(mbed_ts_path, mbed_ts_password);
            led_blue = 1;
            if (EXIT_SUCCESS == status) {
                led_green = 0;
            } else {
                led_red = 0;
            }
            while (1);
        }
    } else {
        pc.printf("Can not obtain DHCP settings\n");
        led_red = 0;
    }
    eth.disconnect();
}
