/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

/**
 * DeviceModelActionArgument
 */
class DeviceModelActionArgument {
    // Instance "variables"/properties...see constructor.

   /**
    * @param {string} A JSON string.
    * @return {DeviceModelActionArgument}
    * @throws Error If the argument type is not one of DeviceModelAttribute.Type.
    */
   static fromJson(jsonObject) {
      // @type {string}
      let name = jsonObject.name != null ? jsonObject.name : 'value';
      // @type {string}
      let type = jsonObject.type;
      let argType = DeviceModelAttribute.getType(type);

      // @type {string}
      let range = jsonObject.range ? jsonObject.range : null;
      // @type {number}
      let min, max;

      if (range) {
         let ranges = range.split(',');
         min = ranges[0];
         max = ranges[1];
      } else {
         min = null;
         max = null;
      }

      let defaultValue = jsonObject.defaultValue;
      return new DeviceModelActionArgument(name, argType, min, max, defaultValue);
   }

    /**
     *
     * @param {string} name
     * @param {string} description
     * @param {DeviceModelAttribute.Type} argType
     * @param {number} lowerBound
     * @param {number} upperBound
     * @param {object} defaultValue
     */
    constructor(name, description, argType, lowerBound, upperBound, defaultValue) {
        // Instance "variables"/properties.
        // @type {string}
        this.name = name;
        // @type {string}
        this.description = description;

        // @type {DeviceModelAttribute.Type}
        this.argType = argType;

        if (this.argType === DeviceModelAttribute.Type.INTEGER ||
            this.argType === DeviceModelAttribute.Type.NUMBER)
        {
            // @type {number}
            this.lowerBound = lowerBound;
            // @type {number}
            this.upperBound = upperBound;
        } else {
            // @type {number}
            this.lowerBound = this.upperBound = null;
        }

        // @type {object}
        this.defaultValue = defaultValue;
    }

    /**
     * The data type of the argument to the action.  If the action does not take an argument, then
     * this method will return {@code null}.
     *
     * @return {DeviceModelAttribute.Type} The action argument's data type, or {@code null}.
     */
    getArgType() {
        return this.argType;
    }

    /**
     * Get the default value of the argument as defined by the device model.  If there is no
     * {@code defaultValue} for the argument in the device model, then this method will return
     * {@code null}.  The value {@code null} is <em>not</em> a default value.
     *
     * @return {object} The default value of the attribute, or {@code null} if no default is
     *         defined.
     */
    getDefaultValue() {
        return null;
    }

    /**
     * For {@link Type#NUMBER} and {@link Type#INTEGER} only, give the lower bound of the acceptable
     * range of values for the action's argument.  {@code null} is always returned for actions other
     * than {@code NUMBER} or {@code INTEGER} type.
     *
     * @return {number} A number or {@code null} if no lower bound has been set.
     */
    getLowerBound() {
        return this.lowerBound;
    }

    /**
     * Get the argument name.
     *
     * @return {string} The action name from the device model.
     */
    getName() {
        return this.name;
    }

    /**
     * For {@link Type#NUMBER} and {@link Type#INTEGER} only, give the upper bound of the acceptable
     * range of values for the action's argument.  {@code null} is always returned for actions other
     * than {@code NUMBER} or {@code INTEGER} type.
     *
     * @return {number} A number, or {@code null} if no upper bound has been set.
     */
    getUpperBound() {
        return this.upperBound;
    }

   /**
    * Returns a string representation of this instance.
    *
    * @return {string} A string  representation of this instance.
    */
   toString() {
      return `DeviceModelActionArgument[name=${this.name}, type=${this.type}, lowerBound=${this.lowerBound}, upperBound=${this.upperBound}, default=${this.default}]`;
   }
}