/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

/**
 * This class represents an action in a device model.
 *
 * @classdesc
 */
class DeviceModelAction {
    // Instance "variables"/properties...see constructor.

    /**
     * Constructs a DeviceModelAction.
     *
     * @param {string} name
     * @param {string} description
     * @param {Array<DeviceModelActionArgument>} args
     * @param {string} alias
     *
     * @class
     */
    constructor(name, description, args, alias) {
        // Instance "variables"/properties.
        // @type {string}
        this.alias = alias;
        // @type {DeviceModelActionArgument[]}
        this.args = args ? args : [];
        // @type {string}
        this.description = description;
        // @type {string}
        this.name = name;
    }


    /**
     * Returns the alias of this action, if any.
     *
     * @return {string} The alias of this action, or <code>null</code>.
     */
    getAlias() {
        return this.alias;
    }

    /**
     * Returns the arguments for this action.
     *
     * return {Array<DeviceModelActionArgument>} The arguments for this action, or
     *        <code>null</code>.
     */
    getArguments() {
        return this.args;
    }

    /**
     * Returns the description of this action.
     *
     * @return {string} The description of this action, or <code>null</code>.
     */
    getDescription() {
        return this.description;
    }

    /**
     * Returns the name of this action.
     *
     * @return {string} The name of this action.
     */
    getName() {
        return this.name;
    }

    /**
     * Returns a string representation of this action.
     *
     * @return {string} A string representation of this action.
     */
    toString() {
       let first = true;
       let argsAsStr;

       this.args.forEach(function(argument) {
          if (!first || (first === false)) {
             argsAsStr += ',' + $port.lineSeparator;
          }

          argsAsStr += '\t' + argument.toString();
       });

       return `DeviceModelAction[name=${this.name}, description=${this.description}, alias=${this.alias}, args=[argsAsStr]]`;
    }
}
