/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

/**
 * An Action to be sent to the server. The action is sent by calling the <code>call()</code> method.
 * <p>
 * The <code>set</code> method returns the <code>Action</code> instance to allow the fields of the
 * action to be set in fluent style.
 * @see VirtualDevice#createAction(String)
 *
 * @class
 * @ignore
 * @private
 */
class Action {
    // Instance "variables"/properties...see constructor.

    /**
     * Returns a DeviceModelAction for the action with the specified name which is associated with
     * the VirtualDevice.
     *
     * @param {iotcs.enterprise.VirtualDevice} virtualDevice - The virtual device associated with
     *        the action.
     * @param {string} actionName - The name of the action.
     * @return {(DeviceModelAction | null)} The device model action or <code>null</code>.
     */
    static getDeviceModelAction(virtualDevice, actionName) {
        // @type {object}
        let deviceModelJson = virtualDevice.getDeviceModel();
        let deviceModel = DeviceModelParser.fromJson(deviceModelJson);
        return deviceModel.getDeviceModelActions().get(actionName);
    }

    /**
     * Constructs an Action.
     *
     * @param {VirtualDevice} virtualDevice - A virtual device.
     * @param {string} actionName - The action name.
     */
    constructor(virtualDevice, actionName) {
        this.deviceModelAction = Action.getDeviceModelAction(virtualDevice, actionName);

        if (!this.deviceModelAction) {
            lib.error("'" + actionName + "' not found.");
        }

        this.actionName = actionName;
        // @type {VirtualDevice}
        this.virtualDevice = virtualDevice;
        this.argumentValues = new Map();
    }

    call() {
        this.virtualDevice.callMultiArgAction(this.actionName, this.getArguments());
    }

    /**
     * Checks the bounds (upper and lower), if there are any bounds, of the value of the argument.
     *
     * @param {DeviceModelActionArgument} argument - The argument to check.
     * @param {number} value - The value of the argument.
     * @throws error If there are bounds for the argument and the value is outside the bounds.
     */
    checkBounds(argument, value) {
        // @type {number}
        const upperBound = argument.getUpperBound();
        // @type {number}
        const lowerBound = argument.getLowerBound();

        // Assumption here is that lowerBound <= upperBound
        if (upperBound != null) {
            if (value > upperBound) {
                lib.error(this.deviceModelAction.getName() + " '" + argument.getName() +
                    "' out of range: " + value + " > " + upperBound);
            }
        }

        if (lowerBound != null) {
            if (value < lowerBound) {
                lib.error(this.deviceModelAction.getName() + " '" + argument.getName() +
                    "' out of range: " + value + " < " + lowerBound);
            }
        }
    }

    /**
     * Returns the argument with the specified name.
     *
     * @param {string} fieldName - The name of the argument.
     * @return {DeviceModelActionArgument} The argument or <code>null</code>.
     */
    getArgument(fieldName) {
        if (!this.deviceModelAction) {
            return null;
        }

        // @type {Array<DeviceModelActionArgument>}
        for (const argument of this.deviceModelAction.getArguments()) {
            if (argument.getName() === fieldName) {
                return argument;
            }
        }

        return null;
    }

    /**
     * Returns the attributes to be updated and their values.
     *
     * @return {Map<string, *>} An object containing the attributes to update.
     */
    getArguments() {
        // @type {object}
        let attributes = new Map();

        // @type {DeviceModelActionArgument[]}
        let args = this.deviceModelAction.getArguments();

        // @type {DeviceModelAction.Argument}
        args.forEach(arg => {
            let name = arg.getName();
            let value = this.argumentValues.get(arg.getName());

            if (!value) {
                value = arg.getDefaultValue();

                if (!value) {
                    lib.error("Missing required argument '" + name + "' to action '" +
                        this.deviceModelAction.getName() + "'");
                }
            }

            attributes.set(name, value);
        });

        return attributes;
    }

    /**
     * Sets the value for the argument with the specified name.
     *
     * @param {string} argumentName - The name of the argument.
     * @param {*} value - The value.
     * @return {Action} An Action for the argument.
     * @throws error If the argument is not in the device model or of is the incorrect type.
     */
    set(argumentName, value) {
        // @type {DeviceModelActionArgument}
        let argument = this.getArgument(argumentName);

        if (!argument) {
            lib.error(argumentName + " not in device model.");
        }

        // @type {string}
        let typeOfValue = typeof value;

        switch (argument.getArgType()) {
            case DeviceModelAttribute.Type.NUMBER:
                if (typeOfValue !== 'number') {
                    lib.error("Value for '" + argumentName + "' is not a NUMBER");
                }

                this.checkBounds(argument, value);
                break;
        case DeviceModelAttribute.Type.INTEGER:
            if (typeOfValue !== 'number') {
                lib.error("Value for '" + argumentName + "' is not an INTEGER");
            }

            this.checkBounds(argument, value);
            break;
        case DeviceModelAttribute.Type.DATETIME:
            if (typeof value.getMonth !== 'function') {
                lib.error("Value for '" + argumentName + "' is not a DATETIME");
            }

            break;
        case DeviceModelAttribute.Type.BOOLEAN:
            if (typeOfValue !== 'boolean') {
                lib.error("value for '" + argumentName + "' is not a BOOLEAN");
            }

            break;
        case DeviceModelAttribute.Type.STRING:
            if (typeOfValue !== 'string') {
                lib.error("value for '" + argumentName + "' is not a STRING");
            }

            break;
        case DeviceModelAttribute.Type.URI:
            if (!(value instanceof lib.ExternalObject)) {
                lib.error("value for '" + argumentName + "' is not an ExternalObject");
            }

            break;
        }

        this.argumentValues.set(argumentName, value);
        return this;
    }
}
