/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

/**
 * Provides for storing and retrieving messages to a persistent store.
 */
class MessagePersistenceImpl {
    // Instance "variables" & properties...see constructor.

    static getInstance() {
        if (lib.oracle.iot.client.device.persistenceEnabled) {
            if (!MessagePersistenceImpl.instance) {
                MessagePersistenceImpl.instance = new MessagePersistenceImpl();
            }

            return MessagePersistenceImpl.instance;
        } else {
            return null;
        }
    }

    constructor() {
        if (lib.oracle.iot.client.device.persistenceEnabled) {
            // Instance "variables" & properties.
            this.TABLE_NAME = 'MESSAGE_PERSISTENT_STORE';
            this.SAVE = 'INSERT INTO ' + this.TABLE_NAME + ' VALUES (?, ?, ?, ?)';
            // This statement is not parameterized.
            this.DELETE = 'DELETE FROM ' + this.TABLE_NAME + ' WHERE uuid = ';
            this.LOAD = 'SELECT * FROM ' + this.TABLE_NAME + ' WHERE ENDPOINT_ID = ? ORDER BY timestamp';
            this.ENDPOINT_ID_INDEX = 'CREATE INDEX endpoint_id ON ' + this.TABLE_NAME + '(ENDPOINT_ID)';
            // Instance "variables" & properties.

            this.db = new sqlite3.Database(lib.oracle.iot.client.device.persistenceDbName, error => {
                if (error) {
                    return console.error(error.message);
                } else {
                    this.createMspsTableIfNotExists();
                }
            });
        }
    }

    /**
     * Creates the message persistent storage table if it doesn't exist.
     */
    createMspsTableIfNotExists() {
        let tableExistsSql = "SELECT name FROM sqlite_master WHERE type='table' AND name=?";

        this.db.get(tableExistsSql, [this.TABLE_NAME], (error, row) => {
            if (error || !row) {
                let maxEndpointIdLength = 100;
                let maxUuidLength = 40;

                let createTableSql =
                    "CREATE TABLE " + this.TABLE_NAME +
                    "(TIMESTAMP BIGINT NOT NULL," +
                    "UUID VARCHAR(" + maxUuidLength + ") NOT NULL," +
                    "ENDPOINT_ID VARCHAR(" + maxEndpointIdLength + ") NOT NULL," +
                    "MESSAGE BLOB," +
                    " PRIMARY KEY (UUID))";

                this.db.run(createTableSql, error => {
                    if (error) {
                        console.log('Error creating table: ' + error);
                    }
                });
            }
        });
    }

    /**
     *
     * @param {Set<Message>} messages
     */
    delete(messages) {
        if (!lib.oracle.iot.client.device.persistenceEnabled) {
            return;
        }

        let stmt = '';

        // Construct multiple delete statements into one for better performance.
        messages.forEach(message => {
            stmt += this.DELETE + "'" + message._.internalObject.clientId + "';";
        });

        if (stmt && (stmt.length > 0)) {
            this.db.exec(stmt);
        }
    }

    /**
     * @param {string} endpointId
     * @return {Promise} - a Set<Message> a set of loaded messages.  May be an empty set if there
     *         are no messages to load.
     */
    load(endpointId) {
        return new Promise((resolve, reject) => {
            let messages = new Set();

            if (!lib.oracle.iot.client.device.persistenceEnabled) {
                resolve(messages);
                return;
            }

            this.db.all(this.LOAD, endpointId, (error, rows) => {
                if (error) {
                    let errorMsg = 'Table does not exist: ' + this.TABLE_NAME;
                    reject(errorMsg);
                } else {
                    rows.forEach(row => {
                        let message = new lib.message.Message();
                        message._.internalObject.clientId = row.UUID;
                        message._.internalObject.eventTime = row.TIMESTAMP;
                        message._.internalObject.source = row.ENDPOINT_ID;
                        let messageJson = JSON.parse(row.MESSAGE);

                        if (messageJson) {
                            message._.internalObject.BASIC_NUMBER_OF_RETRIES =
                                messageJson.BASIC_NUMBER_OF_RETRIES;
                            message._.internalObject.destination = messageJson.destination;
                            message._.internalObject.payload = messageJson.payload;
                            message._.internalObject.priority = messageJson.priority;
                            message._.internalObject.reliability = messageJson.reliability;
                            message._.internalObject.remainingRetries = messageJson.remainingRetries;
                            message._.internalObject.sender = messageJson.sender;
                            message._.internalObject.type = messageJson.type;
                            messages.add(message);
                        }
                    });

                    resolve(messages);
                }
            });
        });
    }

    /**
     *
     * @param {Set<Message>} messages
     * @param {string} endpointId
     */
    save(messages, endpointId) {
        if (!lib.oracle.iot.client.device.persistenceEnabled) {
            return;
        }

        messages.forEach(message => {
            this.db.serialize(function () {
                let stmt = this.prepare(this.SAVE);

                stmt.run(message._.internalObject.eventTime, message._.internalObject.clientId,
                    endpointId, JSON.stringify(message.getJSONObject()), function(error)
                    {
                        if (error) {
                            if (error.message &&
                                !error.message.includes('SQLITE_CONSTRAINT: UNIQUE constraint failed'))
                            {
                                console.log('Error persisting message: ' + error);
                            }
                        }

                        stmt.finalize();
                    });
            });
        });
    }
}
