/**
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * An Action specification, based on a JSON representation of the action specification.
 *
 * @param {string} actionSpec - A JSON string which represents the specification of this action.
 *
 * @class
 * @ignore
 * @private
 */
class ActionSpec extends ActionSpecBase {
    /**
     *
     * @param {string} actionSpec - A JSON string which represents the specification of this action.
     */
    constructor(actionSpec) {
        super(actionSpec);
        this._.onAction = null;
    }

    /**
     * Legacy argument verifier.  Verifies single-argument actions.
     * Verifies that the argument, based on the Action specification, is an argument for the Action.
     *
     * @param {string} argName - The name of the argument to check.
     * @param {*} argValue - The value of the argument to check.
     * @param {VirtualDevice} virtualDevice - The virtual device this argument is for.
     * @param {function({string}, {*}, {VirtualDevice}, {function})} callback - The function to call
     *        back with the results.
     * @returns {*} The original argument if it passes validation, the URI if it's an
     *          ExternalObject, or <code>null</code>.
     *
     * @ignore
     * @private
     */
    validateArgument(argName, argValue, virtualDevice, callback) {
        var isUriCallback = false;

        if (!this.spec.argType) {
            if (typeof argValue !== 'undefined') {
                lib.error('Invalid number of arguments.');
                return;
            }
        } else {
            if (typeof argValue === 'undefined') {
                lib.error('Invalid number of arguments.');
                return;
            }

            if (this.spec.argType === 'URI') {
                if (argValue instanceof lib.ExternalObject) {
                    argValue = argValue.getURI();
                } else if (typeof argValue === 'string') {
                    // get uri from server
                    if (_isStorageCloudURI(argValue)) {
                        isUriCallback = true;

                        virtualDevice.client._.internalDev.createStorageObject(argValue,
                            function (storage, error)
                            {
                                if (error) {
                                    lib.error('Error during creation storage object: ' + error);
                                    return;
                                }

                                var storageObject = new lib.device.StorageObject(storage.getURI(),
                                    storage.getName(), storage.getType(), storage.getEncoding(),
                                    storage.getDate(), storage.getLength());

                                storageObject._.setDevice(virtualDevice.client._.internalDev);
                                storageObject._.setSyncEventInfo(this.spec.name, virtualDevice);

                                if (!_matchType(this.spec.argType, storageObject)) {
                                    lib.error('Type mismatch; action "' + this.spec.name +
                                        '" requires arg type [' + this.spec.argType + '].');

                                    return;
                                }

                                callback(storageObject);   // TODO: djm Do we need to add the argName and virtualDevice here?
                            });

                        return;
                    } else {
                        argValue = new lib.ExternalObject(argValue);
                    }
                } else {
                    lib.error('Invalid URI parameter.');
                    return;
                }
            }

            if (!_matchType(this.spec.argType, argValue)) {
                lib.error('Type mismatch; action "' + this.spec.name + '" requires arg type [' +
                    this.spec.argType + '].');
                return;
            }

            if (this.spec.range &&
                ((argValue < this.spec.range.low) || (argValue > this.spec.range.high)))
            {
                lib.error('Trying to use an argument which is out of range [' +
                    this.spec.range.low + ' - ' + this.spec.range.high + '].');
                return;
            }
        }

        if (!isUriCallback) {
            callback(argName, argValue, virtualDevice, true);
        }
    }

    /**
     * New argument verifier.  Verifies Multiple-argument actions.
     * Verifies that the arguments, based on the Action specification, are arguments for the Action.
     *
     * @param {object[]} args
     * @param {VirtualDevice} virtualDevice
     * @param {callback(object[], VirtualDevice, boolean)} callback
     *
     * @ignore
     * @private
     */
    validateArguments(args, virtualDevice, callback) {
        let newArgs = null;
        let hasUriCallback = false;

        for (let arg of args) {
            let argName = arg.key;
            let argValue = arg.value;
            let argSpec = undefined;

            for (let arg of this.spec.args) {
                if (arg.name === argName) {
                    argSpec = arg;
                    break;
                }
            }

            if (argSpec.type === 'URI') {
                if (argValue instanceof lib.ExternalObject) {
                    argValue = argValue.getURI();
                } else if (typeof arg === 'string') {
                    if (_isStorageCloudURI(argValue)) {
                        hasUriCallback = true;

                        virtualDevice.client._.internalDev.createStorageObject(arg,
                            function (storage, error) {
                                if (error) {
                                    lib.error('Error during creation storage object: ' + error);
                                    return;
                                }

                                let storageObject =
                                    new lib.device.StorageObject(storage.getURI(),
                                        storage.getName(), storage.getType(),
                                        storage.getEncoding(), storage.getDate(),
                                        storage.getLength());

                                storageObject._.setDevice(virtualDevice.client._.internalDev);
                                storageObject._.setSyncEventInfo(argSpec.name, virtualDevice);

                                if (!_matchType(argSpec.type, storageObject)) {
                                    lib.error('Type mismatch for action "' + argSpec.name +
                                        '" requires arg type [' + argSpec.type + '].');

                                    return;
                                }

                                newArgs = newArgs ? newArgs : new {};
                                newArgs.push({'key': argName, 'value': argValue});
                            });
                    } else {
                        argValue = new lib.ExternalObject(argValue);
                    }
                } else {
                    lib.error('Invalid URI parameter.');
                    return;
                }
            }

            if (!_matchType(argSpec.type, argValue)) {
                lib.error('Type mismatch for action "' + argSpec.name + '," requires arg type [' +
                    argSpec.type + '].');
                return;
            }

            if (argSpec.range &&
                ((argValue < argSpec.range.low) || (argValue > argSpec.range.high)))
            {
                lib.error('Trying to use an argument which is out of range: [' + argSpec.range.low +
                    ' - ' + argSpec.range.high + '].');
                return;
            }

            newArgs = (newArgs !== null) ? newArgs : [];
            newArgs.push({'key': argName, 'value': argValue});
        }

        if (!hasUriCallback) {
            callback(args, virtualDevice, true);
        }
    }
}
