/**
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 *
 */

var forge = require('node-forge');
var readline = require("readline");
var net = require('net');
var url = require('url');
var dcl = require('device-library.node.js');
var ecl = require('enterprise-library.node.js');
dcl = dcl({debug:true});
ecl = ecl({debug:true});

var cliArguments = process.argv.slice(2);
var options = [];
var parameters = [];
var optionsIndex = 0;

var typeFlag = 1;
var verbose = false;
var serverUrl = null;
var clientId = null;
var sharedSecret = null;
var password = null;
var connectedDevices = {};

var protocol = 'https';
var host = null;
var port = 443;

var trustAnchors = null;

var trustStoreFile = null;

cliArguments.forEach(function (argument, index) {
    if (['-e', '-h', '-l', '-u', '-v'].indexOf(argument) > -1) {
        options.push(argument);
        optionsIndex = index;
    }
});

cliArguments.forEach(function (argument, index) {
    if ((options.length === 0) || (index > optionsIndex)) {
        parameters.push(argument);
    }
});

console.log('\r\nWARNING: This tool has been deprecated. Please follow documentation to download provisioning file generated by the Oracle IoT Cloud Service.\r\n');

if (options.indexOf('-h') > -1) {
    console.log( "usage:" );
    console.log( " provisioner.sh/bat [option...] <URI> <ID> <Secret> <password> [<ID> <sharedSecret>]" );
    console.log( " provisioner.sh/bat -l <provisioned file> <password>" );
    console.log( "options:" );
    console.log( " -e : to indicate <ID> is an Enterprise Integration ID" );
    console.log( " -u : to indicate the use of user authentication" );
    console.log( " -h : help" );
    console.log( " -l : to list the contents of the provisioned file" );
    console.log( " -v : to list the trust anchor certificates in X.509 format if set" );
    console.log( "<URI> is entered as [<scheme>://]<host>[:<port>]" );
    console.log( "<ID> is either a device Activation ID or Enterprise Integration ID" );
    console.log( "<Secret> is the shared secret from registering the device on the sever or from creating the Enterprise Integration" );
    console.log( "<password> is a passphrase used to protect the integrity of the trusted assets store" );
    console.log( "[<ID> <sharedSecret>] provisions an indirectly connected device:" );
    console.log( "  <ID> is the Activation ID of an indirectly connected device." );
    console.log( "  <sharedSecret> is the shared secret entered when registering the indirectly connected device." );
    console.log( "  The <ID> is separated from the <sharedSecret> by a space and there may be zero or more [<ID> <sharedSecret>] pairs." );
    console.log( "environment variables:" );
    console.log( " NODE_PATH  - points to where all needed node modules are installed" );
    console.log( " SERVER_ROOT_CERTIFICATE  - points to where the root certificate of the server is saved in pem format; used in case of certificate importing failure/warning" );
    process.exit(0);
}

var rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
});

function provisionWithParams(){
    switch (typeFlag) {
        case 1:
            dcl.device.TrustedAssetsManager.provision(clientId + '.json', password, (protocol !== 'https' ? protocol : null),
                host, port, clientId, sharedSecret, trustAnchors, connectedDevices);
            console.log('Created trusted assets store: ' + clientId + '.json');
            break;
        case 2:
            ecl.enterprise.TrustedAssetsManager.provision('trustedAssetsStore.json', password, host, port, null, null, null);
            console.log('Created trusted assets store: trustedAssetsStore.json');
            break;
        case 4:
            ecl.enterprise.TrustedAssetsManager.provision(clientId + '.json', password, host, port, clientId, sharedSecret, trustAnchors);
            console.log('Created trusted assets store: ' + clientId + '.json');
            break;
    }
}

function listTrustContents(){
    try {
        if(((trustStoreFile.indexOf('.json') > -1) && JSON.parse(dcl.$port.file.load(trustStoreFile)).clientId) || (trustStoreFile.indexOf('.json') < 0)){
            dcl.oracle.iot.tam.store = trustStoreFile;
            dcl.oracle.iot.tam.storePassword = password;
            var deviceTam = new dcl.device.TrustedAssetsManager();
            console.log('Trusted assets store: '+trustStoreFile);
            console.log('Server host: '+deviceTam.getServerHost());
            console.log('Server port: '+deviceTam.getServerPort());
            console.log('Activation ID: '+deviceTam.getClientId());
            var endpointId = null;
            try {
                endpointId = deviceTam.getEndpointId();
            } catch (e) {

            }
            var publicKey = null;
            try {
                publicKey = deviceTam.getPublicKey();
            } catch (e) {

            }
            if (endpointId) {
                console.log('Endpoint ID is set.');
            } else {
                console.log('Endpoint ID is not set.');
            }
            if (publicKey) {
                console.log('Public key is set.');
            } else {
                console.log('Public key is not set.');
            }
            if (deviceTam.getTrustAnchorCertificates() && (deviceTam.getTrustAnchorCertificates().length > 0)) {
                console.log('Trust anchor certificates are set.');
                if (verbose) {
                    deviceTam.getTrustAnchorCertificates().forEach(function (cert) {
                        try {
                            console.log(JSON.stringify(forge.pki.certificateFromPem(cert), function (key, value) {
                                var nonPrintable = false;
                                if (typeof value === 'string') {
                                    for (var i = 0; i < value.length; i++) {
                                        if (value.charCodeAt(i) < 0x20 || value.charCodeAt(i) > 0x7E) {
                                            nonPrintable = true;
                                            break;
                                        }
                                    }
                                }
                                if (nonPrintable) {
                                    return forge.util.bytesToHex(value);
                                }
                                return value;
                            }, 4));
                        } catch (e) {
                            console.log(cert);
                        }
                    });
                }
            } else {
                console.log('Trust anchor certificates are not set.');
            }
            if (deviceTam.getConnectedDevices() && Object.keys(deviceTam.getConnectedDevices()).length > 0) {
                console.log('Connected devices: ' + JSON.stringify(deviceTam.getConnectedDevices()));
            } else {
                console.log('Connected devices are not set.');
            }
        } else {
            ecl.oracle.iot.tam.store = trustStoreFile;
            ecl.oracle.iot.tam.storePassword = password;
            var eTam = new ecl.enterprise.TrustedAssetsManager();
            console.log('Trusted assets store: '+trustStoreFile);
            console.log('Server host: '+eTam.getServerHost());
            console.log('Server port: '+eTam.getServerPort());
        }
    } catch (e) {
        console.log('There was an error listing file contents');
        process.exit(1);
    }
}

function getPassword(forList){
    if (!password) {
        rl.question('Enter the passphrase: ', function (answer) {
            if (answer.trim() === '') {
                console.log('Please enter a valid password');
                getPassword();
                return;
            }
            password = answer;
            rl.close();
            if (forList) {
                listTrustContents();
            } else {
                provisionWithParams();
            }
        });
    } else {
        rl.close();
        if (forList) {
            listTrustContents();
        } else {
            provisionWithParams();
        }
    }
}

function getSharedSecret(){
    if ((typeFlag === 1) || (typeFlag === 4)) {
        if (!sharedSecret) {
            rl.question('Enter the shared secret: ', function (answer) {
                if (answer.trim() === '') {
                    console.log('Please enter a valid shared secret');
                    getSharedSecret();
                    return;
                }
                sharedSecret = answer;
                getPassword();
            });
        } else {
            getPassword();
        }
    } else {
        getPassword();
    }
}

function getClientId(){
    if ((typeFlag === 1) || (typeFlag === 4)) {
        if (!clientId) {
            rl.question('Enter the ID: ', function (answer) {
                if (answer.trim() === '') {
                    console.log('Please enter a valid client id');
                    getClientId();
                    return;
                }
                clientId = answer;
                if (dcl.$port.file.exists(clientId + '.json')) {
                    console.log('Trust store file already exists');
                    process.exit(1);
                }
                getSharedSecret();
            });
        } else {
            getSharedSecret();
        }
    } else {
        getSharedSecret();
    }
}

function importCerts() {
    if (host.endsWith('oraclecloud.com') || (typeFlag === 2)) {
        getClientId();
    } else if (process.env.SERVER_ROOT_CERTIFICATE) {
        trustAnchors = dcl.$port.file.load(process.env.SERVER_ROOT_CERTIFICATE)
            .split(/\-{5}(?:B|E)(?:[A-Z]*) CERTIFICATE\-{5}/)
            .filter(function(elem) { return ((elem.length > 1) && (elem.indexOf('M') > -1)); })
            .map(function(elem) { return '-----BEGIN CERTIFICATE-----' + elem.replace(new RegExp('\r\n', 'g'),'\n') + '-----END CERTIFICATE-----'; });
        getClientId();
    } else {
        var noError = false;
        var notSelfSigned = true;
        var socket = new net.Socket();
        var client = forge.tls.createConnection({
            server: false,
            verify: function(connection, verified, depth, certs) {
                if (certs[certs.length-1].issuer.hash === certs[certs.length-1].subject.hash) {
                    notSelfSigned = false;
                }
                trustAnchors = [forge.pki.certificateToPem(certs[certs.length-1]).replace(new RegExp('\r\n', 'g'),'\n')];
                console.log('Server certificate imported');
                noError = true;
                return true;
            },
            connected: function(connection) {
                client.prepare('GET / HTTP/1.0\r\n\r\n');
            },
            tlsDataReady: function(connection) {
                var data = connection.tlsData.getBytes();
                socket.write(data, 'binary');
            },
            dataReady: function(connection) {},
            closed: function() {},
            error: function(connection, error) {
                noError = false;
                console.log('Error when importing certificates with error: '+error.message);
                process.exit(1);
            }
        });
        socket.on('connect', function() {
            client.handshake();
        });
        socket.on('error', function(error) {
            noError = false;
            console.log('Error when importing certificates with error: '+error.message);
            process.exit(1);
        });
        socket.on('data', function(data) {
            client.process(data.toString('binary'));
        });
        socket.on('end', function() {
            if (noError) {
                if (notSelfSigned) {
                    console.log('The certificate imported is not a root certificate.\nYou should set environment SERVER_ROOT_CERTIFICATE to point to the root certificate pem file.');
                }
                getClientId();
            } else {
                console.log('Error when importing certificates');
                process.exit(1);
            }
        });
        socket.connect(port, host);
    }
}

function getServerUrl(){
    if (!serverUrl) {
        rl.question('Enter Cloud Service URI: ', function (answer) {
            serverUrl = answer;
            getServerUrl();
        });
    } else {
        var urlObj = url.parse(serverUrl);
        host = urlObj.hostname;
        if (urlObj.port) {
            try {
                port = parseInt(urlObj.port);
            } catch (e) {

            }
        }
        if ((typeFlag === 1) && urlObj.protocol && (urlObj.protocol.indexOf('mqtt') > -1)) {
            protocol = 'mqtt';
        }
        importCerts();
    }
}

function getTrustStoreFile(){
    if (!trustStoreFile) {
        rl.question('Enter the trust store file: ', function (answer) {
            if (!dcl.$port.file.exists(answer)) {
                console.log('Please enter a valid file that exists');
                getTrustStoreFile();
                return;
            }
            trustStoreFile = answer;
            getPassword(true);
        });
    } else {
        getPassword(true);
    }
}

function parseParams(){
    var n = 0;
    var cur_icd;
    parameters.forEach(function (element, index) {
        switch (index) {
            case 0:
                switch (typeFlag) {
                    case 1:
                    case 2:
                    case 4:
                        serverUrl = element;
                        break;
                    case 3:
                        trustStoreFile = element;
                        break;
                }
                break;
            case 1:
                switch (typeFlag) {
                    case 1:
                    case 4:
                        clientId = element;
                        if (dcl.$port.file.exists(clientId + '.json')) {
                            console.log('Trust store file already exists');
                            process.exit(1);
                        }
                        break;
                    case 2:
                    case 3:
                        password = element;
                        break;
                }
                break;
            case 2:
                switch (typeFlag) {
                    case 1:
                    case 4:
                        sharedSecret = element;
                        break;
                }
                break;
            case 3:
                switch (typeFlag) {
                    case 1:
                    case 4:
                        password = element;
                        break;
                }
                break;
            case 4 + n:
                switch (typeFlag) {
                    case 1:
                        cur_icd = element;
                        break;
                }
                break;
            case 5 + n:
                switch (typeFlag) {
                    case 1:
                        connectedDevices[cur_icd] = element;
                        n += 2;
                        break;
                }
                break;
        }
    });
    switch (typeFlag) {
        case 1:
        case 2:
        case 4:
            getServerUrl();
            break;
        case 3:
            getTrustStoreFile();
            break;
    }
}

if ((options.indexOf('-l') > -1) || (options.indexOf('-v') > -1)) {
    typeFlag = 3;
    if (options.indexOf('-v') > -1) {
        verbose = true;
    }
    if (options.length > 2) {
        console.log('Invalid options for -l');
        process.exit(1);
    }
} else if (options.indexOf('-e') > -1) {
    typeFlag = 4;
    if (options.indexOf('-u') > -1) {
        console.log('Invalid options for -e');
        process.exit(1);
    }
} else if (options.indexOf('-u') > -1) {
    typeFlag = 2;
    if (options.indexOf('-e') > -1) {
        console.log('Invalid options for -u');
        process.exit(1);
    }
}

parseParams();
