/**
 * Copyright (c) 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * The base class for an Action specification, based on a JSON representation of the action
 * specification.
 *
 * @class
 * @ignore
 * @private
 */
class ActionSpecBase {
    /**
     * Constructs an ActionSpecBase class.
     *
     * @param {string} actionSpec - A JSON string which represents the specification of this action.
     */
    constructor(actionSpec) {
        _mandatoryArg(actionSpec, 'object');

        if (!actionSpec.name) {
            lib.error('The attribute specification in the device model is incomplete.');
            return;
        }

        this.spec = {
            alias: (actionSpec.alias || null),
            args: undefined,   // New arguments.
            argType: undefined,   // Legacy argument.
            description: (actionSpec.description || ''),
            name: actionSpec.name,
            range: undefined   // Legacy range.
        };

        /**
         * The arguments for the action.
         *
         * @type {object[]}
         */
        let args = [];

        // Do we have legacy or new action arguments?
        if (actionSpec.argType) {
            // For legacy action arguments.
            this.spec.range =
                actionSpec.range ? _parseRange(actionSpec.argType, actionSpec.range) : null;
            this.spec.argType = (actionSpec.argType || null);
        } else if (actionSpec.arguments) {
            // For new, multiple action arguments.
            actionSpec.arguments.forEach(function(actionArgument) {
                args.push(actionArgument);
            });

            this.spec.args = args;
        } else {
            this.spec.args = null;
        }

        /** @private */
        Object.defineProperty(this, '_', {
            enumerable: false,
            configurable: false,
            writable: true,
            value: {}
        });

        /**
         * @memberof iotcs.ActionSpecBase
         * @member {string} name - the name of this action
         */
        Object.defineProperty(this, 'name', {
            enumerable: true,
            configurable: false,
            writable: false,
            value: this.spec.name
        });

        /**
         * @memberof iotcs.ActionSpecBase
         * @member {string} description - the description of this action
         */
        Object.defineProperty(this, 'description', {
            enumerable: true,
            configurable: false,
            writable: false,
            value: this.spec.description
        });

        /**
         * @memberof iotcs.ActionSpecBase
         * @member {function(object)} onAction - The action to perform when the response to an
         *         execute is received from the other party.
         */
        Object.defineProperty(this, 'onAction', {
            enumerable: false,
            configurable: false,
            get: function () {
                return this._.onAction;
            },
            set: function (newValue) {
                if (!newValue || (typeof newValue !== 'function')) {
                    lib.error('Trying to set something to onAction that is not a function!');
                    return;
                }
                this._.onAction = newValue;
            }
        });
    }
}
