/**
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * The Controller class executes actions and updates attributes on virtual devices by invoking HTTP
 * requests on the server.
 *
 * @class
 * @ignore
 */

/**
 * Constructs a Controller.
 *
 * @param {AbstractVirtualDevice} device - The device associated with this controller.
 */
$impl.Controller = function (device) {
    _mandatoryArg(device, lib.AbstractVirtualDevice);
    this.device = device;
    this.requestMonitors = {};
};

/**
 * Updates the specified attributes by checking the attributes against the device model and sending
 * a message to the server to update the attributes.
 *
 * @param {object} attributeNameValuePairs - An object containing attribute names with associated
 *        attribute values. e.g. { name1:value1, name2:value2, ...}.
 * @param {boolean} [singleAttribute] - Indicates if one attribute needs to be updated. Could be
 *        omitted, in which case the value is false.
 *
 * @function updateAttributes
 * @memberof iotcs.util.Controller.prototype
 */
$impl.Controller.prototype.updateAttributes = function (attributeNameValuePairs, singleAttribute) {
    _mandatoryArg(attributeNameValuePairs, 'object');

    if (Object.keys(attributeNameValuePairs).length === 0) {
        return;
    }
    for(var attributeName in attributeNameValuePairs) {
        if (!this.device[attributeName]) {
            lib.error('device model attribute mismatch');
            return;
        }
    }
    var endpointId = this.device.getEndpointId();
    var deviceModelUrn = this.device.getDeviceModel().urn;
    var selfDevice = this.device;
    var self = this;

    _checkIfDeviceIsDeviceApp(this.device, function () {

        $impl.https.bearerReq({
            method: (singleAttribute ? 'PUT' : 'POST'),
            headers: (singleAttribute ? {} : {
                'X-HTTP-Method-Override': 'PATCH'
            }),
            path: $impl.reqroot
            + '/apps/' + self.device.client.appid
            + ((self.device._.isDeviceApp === 2) ? '/deviceApps/' : '/devices/') + endpointId
            + '/deviceModels/' + deviceModelUrn
            + '/attributes'
            + (singleAttribute ? ('/'+Object.keys(attributeNameValuePairs)[0]) : '')
        }, (singleAttribute ? JSON.stringify({value: attributeNameValuePairs[Object.keys(attributeNameValuePairs)[0]]}) : JSON.stringify(attributeNameValuePairs)), function (response, error) {
            if (!response || error || !(response.id)) {
                _attributeUpdateResponseProcessor(null, selfDevice, attributeNameValuePairs, lib.createError('invalid response on update async request', error));
                return;
            }
            var reqId = response.id;

            try {
                self.requestMonitors[reqId] = new $impl.AsyncRequestMonitor(reqId, function (response, error) {
                    _attributeUpdateResponseProcessor(response, selfDevice, attributeNameValuePairs, error);
                }, self.device.client._.internalClient);
                self.requestMonitors[reqId].start();
            } catch (e) {
                _attributeUpdateResponseProcessor(null, selfDevice, attributeNameValuePairs, lib.createError('invalid response on update async request', e));
            }
        }, function () {
            self.updateAttributes(attributeNameValuePairs, singleAttribute);
        }, self.device.client._.internalClient);
    });
};

/**
 * Invokes the action specified in actionName with a single argument specified in arg.
 *
 * @function invokeSingleArgAction
 * @memberof iotcs.util.Controller.prototype
 *
 * @param {string} actionName The name of the action to invoke.
 * @param {*} [args] - The argument to pass for action execution.  The arguments are specific
 *        to the action.  The description of the argument is provided in the device model.
 */
$impl.Controller.prototype.invokeSingleArgAction = function (actionName, arg) {
    _mandatoryArg(actionName, 'string');

    if (!this.device[actionName]) {
        lib.error('Action: "' + actionName + '" not found in the device model.');
        return;
    }

    let checkedValue;

    // If the action has no argument, checkedValue will still be undefined after this check.
    if ((checkedValue = this.device[actionName].checkAndGetVarArg(arg)) === null) {
        lib.error('Invalid parameters on call to action: "' + actionName + '".');
        return;
    }

    let self = this;
    let endpointId = self.device.getEndpointId();
    let deviceModelUrn = self.device.getDeviceModel().urn;
    let selfDevice = self.device;

    _checkIfDeviceIsDeviceApp(self.device, function () {
        $impl.https.bearerReq({
            method: 'POST',
            path: $impl.reqroot
            + '/apps/' + self.device.client.appid
            + ((self.device._.isDeviceApp === 2) ? '/deviceApps/' : '/devices/') + endpointId
            + '/deviceModels/' + deviceModelUrn
            + '/actions/' + actionName
        }, ((typeof checkedValue !== 'undefined') ? JSON.stringify({value: checkedValue}) : JSON.stringify({})) ,
        function (response, error) {
            if (!response || error || !(response.id)) {
                _actionExecuteResponseProcessor(response, selfDevice, actionName, lib.createError('Invalid response on execute async request.', error));
                return;
            }

            let reqId = response.id;

            try {
                self.requestMonitors[reqId] = new $impl.AsyncRequestMonitor(reqId, function (response, error) {
                    _actionExecuteResponseProcessor(response, selfDevice, actionName, error);
                }, self.device.client._.internalClient);
                self.requestMonitors[reqId].start();
            } catch (e) {
                _actionExecuteResponseProcessor(response, selfDevice, actionName, lib.createError('Invalid response on execute async request.', e));
            }
        }, function () {
            self.invokeSingleArgAction(actionName, checkedValue);
        }, self.device.client._.internalClient);
    });
};

/**
 * Invokes the action specified in actionName with multiple arguments specified in args.
 *
 * @function invokeMultiArgAction
 * @memberof iotcs.util.Controller.prototype
 *
 * @param {string} actionName The name of the action to invoke.
 * @param {Map<string, string>} [args] - A <code>Map</code> of action argument names to action
 *        argument values to pass for action execution.  The arguments are specific to the
 *        action.  The description of the arguments is provided in the device model.
 */
$impl.Controller.prototype.invokeMultiArgAction = function (actionName, args) {
    _mandatoryArg(actionName, 'string');

    if (!this.device[actionName]) {
        lib.error('Action: "' + actionName + '" not found in the device model.');
        return;
    }

    // @type {*}
    let checkedArgs;

    if ((checkedArgs = this.device[actionName].checkAndGetVarArgs(args)) === null) {
        lib.error('Invalid parameters on call to action: "' + actionName + '".');
        return;
    }

    let self = this;
    let endpointId = self.device.getEndpointId();
    let deviceModelUrn = self.device.getDeviceModel().urn;
    let selfDevice = self.device;

    _checkIfDeviceIsDeviceApp(self.device, function () {
        $impl.https.bearerReq({
                method: 'POST',
                path: $impl.reqroot
                + '/apps/' + self.device.client.appid
                + ((self.device._.isDeviceApp === 2) ? '/deviceApps/' : '/devices/') + endpointId
                + '/deviceModels/' + deviceModelUrn
                + '/actions/' + actionName
            }, ((typeof checkedArgs !== 'undefined') ? JSON.stringify(checkedArgs) : JSON.stringify({})),
            function (response, error) {
                if (!response || error || !(response.id)) {
                    _actionExecuteResponseProcessor(response, selfDevice, actionName,
                        lib.createError('Invalid response on execute async request.', error));

                    return;
                }

                let reqId = response.id;

                try {
                    self.requestMonitors[reqId] = new $impl.AsyncRequestMonitor(reqId,
                        function (response, error)
                    {
                        _actionExecuteResponseProcessor(response, selfDevice, actionName, error);
                    }, self.device.client._.internalClient);
                    self.requestMonitors[reqId].start();
                } catch (e) {
                    _actionExecuteResponseProcessor(response, selfDevice, actionName,
                        lib.createError('Invalid response on execute async request.', e));
                }

            }, function () {
                self.invokeMultiArgAction(actionName, checkedArgs);
            }, self.device.client._.internalClient);
    });
};

/**
 * @TODO MISSING DESCRIPTION
 *
 * @memberof iotcs.util.Controller.prototype
 * @function close
 */
$impl.Controller.prototype.close = function () {
    for(var key in this.requestMonitors) {
        this.requestMonitors[key].stop();
    }
    this.requestMonitors = {};
    this.device = null;
};

//////////////////////////////////////////////////////////////////////////////

/**@ignore*/
function _attributeUpdateResponseProcessor (response, device, attributeNameValuePairs, extError) {
    var error = false;
    if (!response || extError) {
        error = true;
        response = extError;
    } else {
        error = (response.status === 'FAILED'
        || (!response.response)
        || (!response.response.statusCode)
        || (response.response.statusCode > 299)
        || (response.response.statusCode < 200));
    }
    var attrObj = {};
    var newValObj = {};
    var tryValObj = {};
    for (var attributeName in attributeNameValuePairs) {
        var attribute = device[attributeName];
        attribute._.onUpdateResponse(error);
        attrObj[attribute.id] = attribute;
        newValObj[attribute.id] = attribute.value;
        tryValObj[attribute.id] = attributeNameValuePairs[attributeName];
        if (error && attribute.onError) {
            var onAttributeErrorTuple = {
                attribute: attribute,
                newValue: attribute.value,
                tryValue: attributeNameValuePairs[attributeName],
                errorResponse: response
            };
            attribute.onError(onAttributeErrorTuple);
        }
    }
    if (error && device.onError) {
        var onDeviceErrorTuple = {
            attributes: attrObj,
            newValues: newValObj,
            tryValues: tryValObj,
            errorResponse: response
        };
        device.onError(onDeviceErrorTuple);
    }
}

/**@ignore*/
function _actionExecuteResponseProcessor(response, device, actionName, error) {
    var action = device[actionName];
    if (action.onAction) {
        action.onAction(response, error);
    }
}

/** @ignore */
function _checkIfDeviceIsDeviceApp(virtualDevice, callback) {

    if (virtualDevice._.isDeviceApp) {
        callback();
        return;
    }

    var deviceId = virtualDevice.getEndpointId();

    var filter = new lib.enterprise.Filter();
    filter = filter.eq('id',deviceId);

    $impl.https.bearerReq({
        method: 'GET',
        path:   $impl.reqroot
        + (virtualDevice.client.appid ? ('/apps/' + virtualDevice.client.appid) : '')
        + '/deviceApps'
        + '?fields=type'
        + '&q=' + filter.toString()
    }, '', function (response, error) {
        if (!response || error || !response.items || !Array.isArray(response.items)) {
            lib.createError('invalid response on device app check request - assuming virtual device is a device');
        } else {
            if ((response.items.length > 0) && response.items[0].type && (response.items[0].type === 'DEVICE_APPLICATION')) {
                virtualDevice._.isDeviceApp = 2;
                callback();
                return;
            }
        }

        virtualDevice._.isDeviceApp = 1;
        callback();

    }, function () {
        _checkIfDeviceIsDeviceApp(virtualDevice, callback);
    }, virtualDevice.client._.internalClient);

}
