/**
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * An Action specification, based on a JSON representation of the action specification.
 *
 * @class
 * @ignore
 * @private
 */
class ActionSpec extends ActionSpecBase {
    /**
     * Constructs an ActionSpec.
     *
     * @param {string} actionSpec - A JSON string which represents the specification of this action.
     */
    constructor(actionSpec) {
        super(actionSpec);
        this._.onAction = function(arg) {};
    }

    /**
     * Verifies that the argument, based on the Action specification in the device model, is an
     * argument for the Action.
     *
     * @param {*} arg The argument to check.
     * @returns {*} The original argument if it passes validation, the URI if it's an ExternalObject,
     *          or <code>null</code>.
     *
     * @ignore
     * @private
     */
    checkAndGetVarArg(arg) {
        if (!this.spec.argType) {
            if (typeof arg !== 'undefined') {
                lib.error('Invalid number of arguments.');
                return null;
            }
        } else {
            if (typeof arg === 'undefined') {
                lib.error('Invalid number of arguments.');
                return null;
            }

            if (this.spec.argType === 'URI') {
                if (arg instanceof lib.ExternalObject) {
                    arg = arg.getURI();
                } else if (typeof arg === 'string') {
                    // nothing to do
                } else {
                    lib.error('Invalid URI parameter.');
                    return null;
                }
            }

            if (!_matchType(this.spec.argType, arg)) {
                lib.error('Type mismatch; action "' + this.spec.name + '" requires arg type [' +
                    this.spec.argType + '].');
                return null;
            }

            if (this.spec.range && ((arg < this.spec.range.low) || (arg > this.spec.range.high))) {
                lib.error('Trying to use an argument which is out of range [' +
                    this.spec.range.low + ' - ' + this.spec.range.high + '].');
                return null;
            }
        }
        return arg;
    };

    /**
     * Verifies that the arguments, based on the Action specification in the device model, are
     * arguments for the Action.
     *
     * @param {Map<string, string>} [args] - A <code>Map</code> of action argument names to action
     *        argument values to pass for action execution.  The arguments are specific to the
     *        action.  The description of the arguments is provided in the device model.
     * @return {object} The original arguments, as an object (suitable to JSON.stringify), if they
     *          pass validation or <@code>null</code>.  If an ExternalObject is supplied, it's URI
     *          is stored in the returned args.
     *
     * @ignore
     * @private
     */
    checkAndGetVarArgs(args) {
        if (this.spec.args.length === args.size) {
            // New action arguments
            let newArgs = null;
            let self = this;

            args.forEach(function(argValue, argName) {
                if (typeof argValue === 'undefined') {
                    lib.error('Invalid number of arguments.');
                    return null;
                }

                let argSpec = undefined;

                for (let aSpec of self.spec.args) {
                    if (aSpec.name === argName) {
                        argSpec = aSpec;
                        break;
                    }
                }

                if (argSpec.type === 'URI') {
                    if (argValue instanceof lib.ExternalObject) {
                        argValue = argValue.getURI();
                    } else if (typeof argValue === 'string') {
                        // nothing to do
                    } else {
                        lib.error('Invalid URI parameter.');
                        return null;
                    }
                }

                if (!_matchType(argSpec.type, argValue)) {
                    lib.error('Type mismatch for action "' + self.spec.name +
                        '" requires arg type [' + argSpec.type + '].');

                    return null;
                }

                if (argSpec.range &&
                    ((argValue < argSpec.range.low) || (argValue > argSpec.range.high)))
                {
                    lib.error('Trying to use an argument which is out of range [' +
                        argSpec.range.low + ' - ' + argSpec.range.high + '].');

                    return null;
                }

                newArgs = newArgs ? newArgs : {};
                newArgs[argName] = argValue;
            });

            return newArgs;
        } else {
            lib.error('Invalid number of arguments.');
            return null;
        }
    }
}
