/**
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * DeviceAnalog combines a device (endpoint id and attributes) with a model.
 *
 * @class
 * @ignore
 * @private
 */
class DeviceAnalog {
    // Instance "variables" & properties...see constructor.

    /**
     *
     * @param {$impl.DirectlyConnectedDevice} directlyConnectedDevice
     * @param {DeviceModel} deviceModel
     * @param {string} endpointId the endpoint ID of the DirectlyConnectedDevice with this device
     * model.
     */
    constructor(directlyConnectedDevice, deviceModel, endpointId) {
        // Instance "variables" & properties.
        /**
         *
         * @type {Map<String, Object>}
         */
        this.attributeValueMap = new Map();
        /**
         *
         * @type {$impl.DirectlyConnectedDevice}
         */
        this.directlyConnectedDevice = directlyConnectedDevice;
        /**
         *
         * @type {DeviceModel}
         */
        this.deviceModel = deviceModel;
        /**
         *
         * @type {string}
         */
        this.endpointId = endpointId;
        // Instance "variables" & properties.
    }

    /**
     * Invoke an action. The {@code argumentValues} parameter may be empty if there are no arguments
     * to pass, but will not be {@code null}.
     *
     * @param {string} actionName The name of the action to call.
     * @param {Map<string, ?>} argumentValues  The data to pass to the action, possibly an empty
     *        Map but never {@code null}.
     * @throws {lib.error} If
     */
    call(actionName, argumentValues) {
        // @type {Map<string, DeviceModelAction}
        const deviceModelActionMap = this.deviceModel.getDeviceModelActions();

        if (!deviceModelActionMap) {
            return;
        }

        // @type {DeviceModelAction}
        const deviceModelAction = deviceModelActionMap.get(actionName);

        if (!deviceModelAction) {
            return;
        }

        // What this is doing is pushing a request message into the message
        // queue for the requested action. To the LL, it is handled just like
        // any other RequestMessage. But we don't want this RM going to the
        // server, so the source and destination are set to be the same
        // endpoint id. In SendReceiveImpl, if there is a RequestMessage
        // with a source == destination, it is treated it specially.
        // @type {object}
        let requestMessage = {};

        requestMessage.payload = {
            body: '',
            method: 'POST',
            url: "deviceModels/" + this.getDeviceModel().getUrn() + "/actions/" + actionName
        };

        requestMessage.destination = this.getEndpointId();
        requestMessage.source = this.getEndpointId();
        requestMessage.type = lib.message.Message.Type.REQUEST;

        // @type {Array<DeviceModelActionArgument>}
        const argumentList = deviceModelAction.getArguments();

        try {
            // @type {object}
            let actionArgs = null;

            argumentList.forEach (argument => {
                // @type {object}
                let value = argumentValues.get(argument.getName());

                if (!value) {
                    value = argument.getDefaultValue();

                    if (!value) {
                        lib.error("Action not called: missing argument '" + argument.getName() +
                            " in call to '" + actionName + "'");
                        return;
                    }
                }

                this.encodeArg(actionArgs, deviceModelAction, argument, value);
            });

            requestMessage.payload.body = actionArgs.toString();
        } catch (error) {
            lib.log(error.message);
            return;
        }

        // @type {boolean}
        const useLongPolling = lib.oracle.iot.client.device.disableLongPolling;
        // Assumption here is that, if you are using long polling, you are using message dispatcher.
        // This could be a bad assumption. But if long polling is disabled, putting the message on
        // the request buffer will work regardless of whether message dispatcher is used.
        if (useLongPolling) {
            try {
                // @type {Message} (ResponseMessage)
                const responseMessage =
                    new lib.device.util.RequestDispatcher().dispatch(requestMessage);
            } catch (error) {
                console.log(error);
            }
        } else {
            // Not long polling, push request message back on request buffer.
            try {
                // @type {Message} (ResponseMessage)
                const responseMessage =
                    new lib.device.util.RequestDispatcher().dispatch(requestMessage);
            } catch (error) {
                console.log(error);
            }
        }
    }

    /**
     * Determines if value is outside the lower and upper bounds of the argument.
     *
     * @param {DeviceModelAction} deviceModelAction
     * @param {DeviceModelActionArgument} argument
     * @param {number} value
     * @throws {error} if value is less than the lower bound or greater than the upper bound of the
     *         argument.
     */
    checkBounds(deviceModelAction, argument, value) {
        // @type {number}
        const upperBound = argument.getUpperBound();
        // @type {number}
        const lowerBound = argument.getLowerBound();

        // Assumption here is that lowerBound <= upperBound.
        if (upperBound) {
            if (value > upperBound) {
                throw new lib.error(deviceModelAction.getName() + " '" + argument.getName() +
                    "' out of range: " + value + " > " + upperBound);
            }
        }

        if (lowerBound) {
            if(value < lowerBound) {
                throw new lib.error(deviceModelAction.getName() + " '" + argument.getName() +
                    "' out of range: " + value + " < " + lowerBound);
            }
        }
    }

    /**
     * Checks the data type of the value against the device model, converts the value to the
     * required type if needed, and adds the argument/value to the jsonObject.
     *
     * @param {object} jsonObject - The JSON object to add the argument and value to.
     * @param {DeviceModelAction} deviceModelAction - The device model action.
     * @param {DeviceModelActionArgument} argument - The device model action argument specification.
     * @param {object} value - The argument value.
     * @throws lib.error If there was a problem encoding the argument.
     */
    encodeArg(jsonObject, deviceModelAction, argument, value) {
        // @type {string}
        const actionName = deviceModelAction.getName();
        // @type {string}
        const argumentName = argument.getName();
        // @type {string}
        const typeOfValue = typeof value;

        // @type {DeviceModelAttribute.Type}
        switch (argument.getArgType()) {
            case DeviceModelAttribute.Type.NUMBER:
                if (typeOfValue !== 'number') {
                    lib.error(actionName + " value for '" + argument + "' is not a NUMBER");
                }

                this.checkBounds(deviceModelAction, argument, value);
                jsonObject.put(argumentName, value);
                break;
            case DeviceModelAttribute.Type.INTEGER:
                if (typeOfValue !== 'integer') {
                    lib.error(actionName + " value for '" + argumentName + "' is not an INTEGER");
                }

                this.checkBounds(deviceModelAction, argument, value);
                jsonObject.put(argumentName, value);
                break;
            case DeviceModelAttribute.Type.DATETIME:
                if ((typeOfValue !== 'date') && (typeOfValue !== 'long')) {
                    lib.error(actionName + " value for '" + argumentName + "' is not a DATETIME");
                }

                if (typeOfValue === 'date') {
                    let d = new Date();
                    jsonObject.put(argumentName, value.getMilliseconds());
                }

                jsonObject.put(argumentName, value);
                break;
            case DeviceModelAttribute.Type.BOOLEAN:
                if (typeOfValue !== 'boolean') {
                    lib.error(actionName + " value for '" + argumentName + "' is not a BOOLEAN");
                }

                jsonObject.put(argumentName, value);
                break;
            case DeviceModelAttribute.Type.STRING:
                if (typeOfValue !== 'string') {
                    lib.error(actionName + " value for '" + argumentName + "' is not a STRING");
                }

                jsonObject.put(argumentName, value);
                break;
            case DeviceModelAttribute.Type.URI:
                if (!(value instanceof lib.ExternalObject)) {
                    lib.error(actionName + " value for '" + argumentName +
                        "' is not an ExternalObject");
                }

                jsonObject.put(argumentName, value);
                break;
            default:
                lib.error(actionName + " argument '" + argumentName + "' has an unknown type");
        }
    }

    /**
     * Returns the attribute value of the attribute with the specified name.  If the attribute value
     * is not available, returns the attribute's default value.
     *
     * @param {string} attributeName - The name of the attribute.
     * @return {object} The attribute's value or default value.
     */
    getAttributeValue(attributeName) {
        /** {$impl.Attribute} */
        let deviceModelAttribute = this.deviceModel.getDeviceModelAttributes().get(attributeName);

        if (deviceModelAttribute === null) {
            throw new Error(this.deviceModel.getUrn() + " does not contain attribute " +
                attributeName);
        }

        let value = this.attributeValueMap.get(attributeName);

        if (value === null) {
            value = deviceModelAttribute.defaultValue;
        }

        return value;
    }


    /**
     * Returns the device model.
     *
     * @returns {DeviceModel} The device model.
     */
    getDeviceModel() {
        return this.deviceModel;
    }

    /**
     * Returns the device's endpoint ID.
     *
     * @return {string} The device's endpoint ID.
     */
    getEndpointId() {
        return this.directlyConnectedDevice.getEndpointId();
    }

    /**
     * Enqueue's the message for dispatching.
     *
     * @param {Message} message - The message to be enqueued.
     */
    queueMessage(message) {
        try {
            this.directlyConnectedDevice.dispatcher.queue(message);
        } catch(error) {
            console.log('Error queueing message: ' + error);
        }
    }

    /**
     * Set the named attribute to the given value.
     *
     * @param {string} attribute - The attribute to set.
     * @param {object} value - The value of the attribute.
     * @throws Error If the attribute is not in the device model, the value is <code>null</code>, or
     *         the value does not match the attribute type.
     */
    setAttributeValue(attribute, value) {
        if (value === null) {
            throw new Error("value cannot be null");
        }

        let deviceModelAttribute = this.deviceModel.getDeviceModelAttributes().get(attribute);

        if (!deviceModelAttribute) {
            throw new Error(this.deviceModel.getUrn() + " does not contain attribute " + attribute);
        }

        // {DeviceModelAttribute.Type}
        let type = deviceModelAttribute.type;
        let badValue;
        let typeOfValue = null;

        switch (type) {
            // TODO: e don't need all of these types in JavaScript.
            case DeviceModelAttribute.Type.DATETIME:
            case DeviceModelAttribute.Type.INTEGER:
            case DeviceModelAttribute.Type.NUMBER:
                typeOfValue = typeof value === 'number';
                badValue = !typeOfValue;
                break;
            case DeviceModelAttribute.Type.STRING:
            case DeviceModelAttribute.Type.URI:
                typeOfValue = typeof value === 'string';
                badValue = !typeOfValue;
                break;
            case DeviceModelAttribute.Type.BOOLEAN:
                typeOfValue = typeof value === 'boolean';
                badValue = !typeOfValue;
                break;
            default:
                throw new Error('Unknown type ' + type);
        }

        if (badValue) {
            throw new Error("Cannot set '"+ this.deviceModel.getUrn() + ":attribute/" + attribute + "' to " +
                value.toString());
        }

        this.attributeValueMap.set(attribute, value);
    }
}
